import React, { useEffect, useState } from 'react';
import { Head, usePage, router, useForm } from '@inertiajs/react';
import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { toast } from 'react-hot-toast';
import Transactions from '../Admin/Transactions/Index';

const Success = ({ auth, paymentId, status, mchOrderNo, amount, transaction, settings }) => {
    const { flash, errors } = usePage().props;
    const [paymentDetails, setPaymentDetails] = useState(null);
    const [loading, setLoading] = useState(true);

    const { data, setData, post, processing, reset } = useForm({
        payOrderNo: '',
        mchOrderNo: '',
        identifier: ''
    });

    // Show flash messages
    useEffect(() => {
        if (flash.success) {
            toast.success(flash.success);
            reset();
        }
        if (flash.error) {
            toast.error(flash.error);
        }
    }, [flash]);

    useEffect(() => {
        const fetchPaymentDetails = async () => {
            try {
                    // Format amount from cents to dollars
                    const formattedAmount = (amount / 100).toFixed(2);
                    setPaymentDetails({
                        status: status,
                        orderId: mchOrderNo,
                        transaction: transaction,
                        amount: formattedAmount,
                        payOrderNo: paymentId // This is the payOrderNo for GGPayout
                    });
                    setData('mchOrderNo', mchOrderNo);
                    setData('payOrderNo', paymentId);
            } catch (error) {
                console.error('Error fetching payment details:', error);
                toast.error('Failed to fetch payment details');
            } finally {
                setLoading(false);
            }
        };

        fetchPaymentDetails();
    }, [paymentId, status, mchOrderNo, amount, transaction]);

    const handleIdentifierChange = (e) => {
        const value = e.target.value;
        setData('identifier', value);
        
        // If identifier is valid, submit the form
        if (value && value.startsWith('#')) {
            //handleFillOrder();
        }
    };

    const handleFillOrder = () => {
        if (!data.identifier) {
            toast.error('Please enter a transaction identifier');
            return;
        }

        if (!data.identifier.startsWith('#')) {
            toast.error('Identifier must start with #');
            return;
        }

        // Set the other form data
        setData({
            ...data,
            payOrderNo: paymentDetails.payOrderNo,
            mchOrderNo: paymentDetails.orderId
        });

        post('/payment/ecashapp/fill', {
            preserveScroll: true,
            onSuccess: () => {
                reset();
            }
        });
    };

    if (loading) {
        return (
            <AuthenticatedLayout user={auth.user} settings={settings}>
                <div className="flex items-center justify-center min-h-[80vh]">
                    <div className="text-center">
                        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-indigo-600 mx-auto"></div>
                        <p className="mt-4 text-gray-600 dark:text-gray-400">Verifying payment...</p>
                    </div>
                </div>
            </AuthenticatedLayout>
        );
    }
    const UPstatus = status == 'failed' ? 'failed' : paymentDetails?.transaction?.payment_status;
    return (
        <AuthenticatedLayout user={auth.user} settings={settings}>
            <Head title="Payment Status | VegasFish" />
            <div className="flex items-center justify-center min-h-[80vh]">
                <div className="container mx-auto px-4 sm:px-6 lg:px-8 py-5 max-w-2xl">
                    <div className="w-full p-8 bg-white dark:bg-gray-800 rounded-lg shadow-md">
                        <div className="text-center">
                            {(() => {
                                switch(UPstatus) {
                                    case 'completed':
                                        return (
                                            <>
                                                <div className="w-20 h-20 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-6">
                                                    <svg className="w-10 h-10 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7"></path>
                                                    </svg>
                                                </div>
                                                <h1 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">Payment Successful!</h1>
                                                <p className="text-gray-600 dark:text-gray-400 mb-6">Thank you for your payment. Your points will be added to your account shortly.</p>
                                            </>
                                        );
                                    case 'pending':
                                    case 'created':
                                        return (
                                            <>
                                                <div className="w-20 h-20 bg-yellow-100 rounded-full flex items-center justify-center mx-auto mb-6">
                                                    <svg className="w-10 h-10 text-yellow-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                                    </svg>
                                                </div>
                                                <h1 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">Payment Processing</h1>
                                                <p className="text-gray-600 dark:text-gray-400 mb-6">Your payment is being processed. Please wait while we confirm your transaction.</p>
                                            </>
                                        );
                                    case 'failed':
                                        return (
                                            <>
                                                <div className="w-20 h-20 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-6">
                                                    <svg className="w-10 h-10 text-red-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M6 18L18 6M6 6l12 12"></path>
                                                    </svg>
                                                </div>
                                                <h1 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">Payment Failed</h1>
                                                <p className="text-gray-600 dark:text-gray-400 mb-6">Your payment could not be processed. Please try again or contact support.</p>
                                            </>
                                        );
                                    case 'revoked':
                                    case 'refunded':
                                        return (
                                            <>
                                                <div className="w-20 h-20 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-6">
                                                    <svg className="w-10 h-10 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M3 10h10a8 8 0 018 8v2M3 10l6 6m-6-6l6-6"></path>
                                                    </svg>
                                                </div>
                                                <h1 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">{paymentDetails?.status === 'revoked' ? 'Payment Revoked' : 'Payment Refunded'}</h1>
                                                <p className="text-gray-600 dark:text-gray-400 mb-6">Your payment has been {paymentDetails?.status === 'revoked' ? 'revoked' : 'refunded'}. If you have any questions, please contact support.</p>
                                            </>
                                        );
                                    case 'closed':
                                        return (
                                            <>
                                                <div className="w-20 h-20 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-6">
                                                    <svg className="w-10 h-10 text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M6 18L18 6M6 6l12 12"></path>
                                                    </svg>
                                                </div>
                                                <h1 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">Payment Closed</h1>
                                                <p className="text-gray-600 dark:text-gray-400 mb-6">This payment session has been closed. Please start a new payment if needed.</p>
                                            </>
                                        );
                                    default:
                                        return (
                                            <>
                                                <div className="w-20 h-20 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-6">
                                                    <svg className="w-10 h-10 text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M8.228 9c.549-1.165 2.03-2 3.772-2 2.21 0 4 1.343 4 3 0 1.4-1.278 2.575-3.006 2.907-.542.104-.994.54-.994 1.093m0 3h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                                    </svg>
                                                </div>
                                                <h1 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">Unknown Status</h1>
                                                <p className="text-gray-600 dark:text-gray-400 mb-6">The payment status is unknown. Please contact support for assistance.</p>
                                            </>
                                        );
                                }
                            })()}
                            <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-6 mb-6">
                                <div className="grid grid-cols-2 gap-4 text-left">
                                    <div className="text-gray-600 dark:text-gray-400">Order ID:</div>
                                    <div className="font-medium text-gray-900 dark:text-white">{paymentDetails?.orderId}</div>
                                    
                                    <div className="text-gray-600 dark:text-gray-400">Amount:</div>
                                    <div className="font-medium text-gray-900 dark:text-white">
                                        ${paymentDetails?.amount} USD
                                    </div>
                                    
                                    <div className="text-gray-600 dark:text-gray-400">Payment Gateway:</div>
                                    <div className="font-medium text-gray-900 dark:text-white">{paymentDetails?.transaction?.payment_gateway}</div>
                                    
                                    <div className="text-gray-600 dark:text-gray-400">Status:</div>
                                    <div className="font-medium text-gray-900 dark:text-white capitalize">
                                        {paymentDetails?.status}
                                    </div>
                                </div>

                                {/* Add Ecashapp Fill Order Form */}
                                {/*paymentDetails?.transaction?.payment_gateway == 'ggpayout' && paymentDetails.transaction?.payment_method == 'ecashapp' &&
                                 paymentDetails?.transaction?.payment_status == 'pending' && (
                                    <div className="mt-6 pt-6 border-t border-gray-200 dark:border-gray-600">
                                        <h3 className="text-lg font-semibold mb-4">Fill Order Details</h3>
                                        <div className="flex flex-col gap-4">
                                            <div>
                                                <input
                                                    type="text"
                                                    value={data.identifier}
                                                    onChange={handleIdentifierChange}
                                                    placeholder="Enter transaction identifier (e.g., #12345)"
                                                    className={`w-full p-2 border rounded-md dark:bg-gray-600 dark:border-gray-500 ${
                                                        errors.identifier ? 'border-red-500' : ''
                                                    }`}
                                                />
                                                {errors.identifier && (
                                                    <p className="mt-1 text-sm text-red-500">{errors.identifier}</p>
                                                )}
                                                {errors.payOrderNo && (
                                                    <p className="mt-1 text-sm text-red-500">{errors.payOrderNo}</p>
                                                )}
                                                {errors.mchOrderNo && (
                                                    <p className="mt-1 text-sm text-red-500">{errors.mchOrderNo}</p>
                                                )}
                                                {errors.error && (
                                                    <p className="mt-1 text-sm text-red-500">{errors.error}</p>
                                                )}
                                            </div>
                                            <button
                                                onClick={handleFillOrder}
                                                disabled={processing}
                                                className={`px-4 py-2 rounded-md text-white ${
                                                    processing 
                                                        ? 'bg-gray-400 cursor-not-allowed' 
                                                        : 'bg-indigo-600 hover:bg-indigo-700'
                                                }`}
                                            >
                                                {processing ? 'Processing...' : 'Fill Order'}
                                            </button>
                                        </div>
                                        <p className="mt-2 text-sm text-gray-500 dark:text-gray-400">
                                            Enter the transaction identifier starting with # to fill the order
                                        </p>
                                    </div>
                                )*/}
                            </div>

                            <div className="flex justify-center space-x-4">
                                <a
                                    href="/"
                                    className="inline-flex items-center px-6 py-3 border border-transparent text-base font-medium rounded-md text-white bg-[#FF6F00] hover:bg-[#960018] transition-all duration-300"
                                >
                                    Go to Dashboard
                                </a>
                                <a
                                    href="/addpoints"
                                    className="inline-flex items-center px-6 py-3 border border-gray-300 text-base font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 dark:bg-gray-700 dark:text-gray-300 dark:border-gray-600 dark:hover:bg-gray-600 transition-all duration-300"
                                >
                                    Make Another Payment
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AuthenticatedLayout>
    );
};

export default Success;