import React, { useState, useEffect, useRef } from 'react';
import { useForm, router, Head, usePage } from '@inertiajs/react';
import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { SwipeableButton } from "react-swipeable-button";
import HeroImg from "../../../images/hero-img.jpg"
import Coinbase from '../../../images/coinbase.png';
import BitPay from '../../../images/bitpay.png';
import Venmo from '../../../images/venmo.png';
import PayPal from '../../../images/paypal.png';
import Zelle from '../../../images/zelle.png';
import applePay from '../../../images/applepay.png';
import GPay from '../../../images/googlepay.webp';
import CreditCard from '../../../images/card.png';
import CashApp from '../../../images/cashapp.png';
import Exodus from '../../../images/exodus.png';
import { toast } from 'react-hot-toast';
import axios from 'axios';

const PaymentForm = ({ auth, games, flash, settings, walletButtons }) => {
    const pageProps = usePage().props;
    const { data, setData, post, errors } = useForm({
        amount: '',
        game: '',
        username: '',
        wallet: '',
        //currency: 'SATS'
    });
    const DisplayErrors = {
        error: pageProps.errors?.error || flash?.error || null,
        message: pageProps.errors?.message || flash?.message || null,
        data: pageProps.errors?.data || flash?.data || null
    }
    const [copyTimer, setCopyTimer] = useState(0);
    const [selectedGame, setSelectedGame] = useState(null);
    const [formErrors, setFormErrors] = useState({});
    const [paymentMethodOptions, setPaymentMethodOptions] = useState({});
    const [TransactionData, setTransactionData] = useState({});
    const [activeTab, setActiveTab] = useState('lightning');
    const [checkingPayment, setCheckingPayment] = useState(false);
    const [paymentCompleted, setPaymentCompleted] = useState(false);
    const [paymentID, setPaymentID] = useState(null);
    const [message, setMessage] = useState('');
    const [walletOpened, setWalletOpened] = useState(false);
    const [lastAttemptedWallet, setLastAttemptedWallet] = useState(null);
    const [showPaymentGateways, setShowPaymentGateways] = useState(false);
    const [selectedGateway, setSelectedGateway] = useState(null);
    const [processing, setProcessing] = useState(false);
    const [targetCurrency, setTargetCurrency] = useState(null);
    const [baseAmount, setbaseAmount] = useState('');
    const timerRef = useRef(null);
    const [showBitcoinFees, setShowBitcoinFees] = useState(false);

    const swipeableButtonRef = useRef();

    // Updated GGPayout wallet options
    const ggPayoutWallets = [
        { id: 'ecashapp', name: 'Cash App Personal', icon: CashApp },
        { id: 'zelle', name: 'Zelle', icon: Zelle }, // You might want to add a Zelle icon
        /*{ id: 'cashapp', name: 'Cash App', icon: CashApp },
        { id: 'btcpay', name: 'Cash App BTC', icon: BitPay },
        { id: 'paypal', name: 'PayPal', icon: PayPal },
        { id: 'applepay', name: 'Apple Pay', icon: applePay }, // You might want to add an Apple Pay icon
        { id: 'googlepay', name: 'Google Pay', icon: GPay },
        { id: 'card', name: 'Credit Card', icon: CreditCard }*/ // You might want to add a credit card icon
    ];

    const getWalletButtons = (type) => {
        if (!walletButtons) return [];

        let buttons = [];
        
        // Add buttons from the 'both' group
        if (walletButtons.both) {
            buttons = [...walletButtons.both];
        }

        // Add buttons from the specific type group (onchain/lightning)
        const typeKey = type.toLowerCase();
        if (walletButtons[typeKey]) {
            buttons = [...buttons, ...walletButtons[typeKey]];
        }

        // Filter enabled buttons and sort by display order
        return buttons
            .filter(button => button.is_enabled)
            .sort((a, b) => a.display_order - b.display_order);
    };

    useEffect(() => {
        if(DisplayErrors && DisplayErrors?.error) {
            setFormErrors((prev) => ({ ...prev, general: DisplayErrors.error }));
        }

        if (flash && flash.data?.paymentMethodOptions) {
            setPaymentMethodOptions(flash.data?.paymentMethodOptions);
            setTransactionData(flash.data?.TransactionData);
            setPaymentID(flash?.data?.paymentID);
        }
    }, [flash]);

    useEffect(() => {
        let interval;
        if (checkingPayment && !paymentCompleted) {
            interval = setInterval(() => checkPaymentStatus(paymentID, interval), 4000);
        }
        return () => clearInterval(interval);
    }, [checkingPayment, paymentCompleted]);

    const handleReset = () => {
        swipeableButtonRef.current?.buttonReset();
    };

    const handleProceed = async () => {
        setFormErrors({});

        // Convert amount to number for proper comparison
        const amountNum = Number(data.amount);
        const minAmount = Number(settings.min_amount);
        const maxAmount = Number(settings.max_amount);
        
        if (!data.amount || amountNum < minAmount || amountNum > maxAmount) {
            setFormErrors((prev) => ({ 
                ...prev, 
                amount: `Amount must be between ${settings.min_amount} and ${settings.max_amount} ${settings.default_currency}.` 
            }));
            handleReset();
            return;
        }

        if (!selectedGame) {
            setFormErrors((prev) => ({ ...prev, game: "Please select a game." }));
            handleReset();
            return;
        }

        if (!data.username) {
            setFormErrors((prev) => ({ ...prev, username: "Please enter a username." }));
            handleReset();
            return;
        }

        // Show payment gateway selection instead of directly generating address
        setShowPaymentGateways(true);
        handleReset();
    };

    const handleGatewaySelect = async (gateway) => {
        if(paymentID) {
            removeOldTryspeedTransaction(paymentID);
        }

        // Update amount with fees if cashapp fee is set
        if (settings.cashapp_fees && gateway === 'ggpayout') {
            const fee = parseFloat(settings.cashapp_fees);
            const amount = parseFloat(data.amount);
            const amountWithFees = amount + (amount * fee / 100);
            setData('amount', amountWithFees.toFixed(2));
        }else if (settings.usdt_fees && gateway === 'usdt') {
            const fee = parseFloat(settings.usdt_fees);
            const amount = parseFloat(data.amount);
            const amountWithFees = amount + (amount * fee / 100);
            setData('amount', amountWithFees.toFixed(2));
        }else if (settings.usdc_fees && gateway === 'usdc') {
            const fee = parseFloat(settings.usdc_fees);
            const amount = parseFloat(data.amount);
            const amountWithFees = amount + (amount * fee / 100);
            setData('amount', amountWithFees.toFixed(2));
        }else if (settings.bitcoin_fees && gateway === 'tryspeed') {
           
        }else {
            setData('amount', data.amount);
        }

        setSelectedGateway(gateway);
        setData('wallet', '');
        //setData('currency', 'SATS');
        setTargetCurrency(null);
        if (gateway === 'tryspeed') {
            setPaymentMethodOptions({});
            setTransactionData({});
            setPaymentID(null);
            
            const response = await post('/addpoints/generate-address', { 
                amount: data.amount, 
                game: selectedGame,
                username: data.username
            });
    
            if (response && response.errors) {
                setFormErrors(response.props.errors);
            } else {
                if (response?.data?.paymentMethodOptions) {
                    setShowBitcoinFees(false);
                    setPaymentMethodOptions(response?.data?.paymentMethodOptions);                
                } else {
                    //setFormErrors((prev) => ({ ...prev, general: "No payment method options available1." }));
                }
            }

        } else if (gateway === 'usdt') {
            setPaymentMethodOptions({});
            setTransactionData({});
            setPaymentID(null);

            const response = await post('/addpoints/generate-address-usdt', { 
                amount: data.amount, 
                game: selectedGame,
                username: data.username
            });
    
            if (response && response.errors) {
                setFormErrors(response.props.errors);
            } else {
                if (response?.data?.paymentMethodOptions) {
                    setShowBitcoinFees(false);
                    setPaymentMethodOptions(response?.data?.paymentMethodOptions);                
                } else {
                    //setFormErrors((prev) => ({ ...prev, general: "No payment method options available1." }));
                }
            }
        } else if (gateway === 'usdc') {
            setPaymentMethodOptions({});
            setTransactionData({});
            setPaymentID(null);
            
            const response = await post('/addpoints/generate-address-usdc', { 
                amount: data.amount, 
                game: selectedGame,
                username: data.username
            });
            
            if (response && response.errors) {
                setFormErrors(response.props.errors);
            } else {
                if (response?.data?.paymentMethodOptions) {
                    setShowBitcoinFees(false);
                    setPaymentMethodOptions(response?.data?.paymentMethodOptions);                
                } else {
                    //setFormErrors((prev) => ({ ...prev, general: "No payment method options available1." }));
                }
            }
        } else if (gateway === 'payken') {
            handlePaykenPayNow();
        }
    };

    /*const handleCurrencySelect = async () => {
        if(paymentID) {
            removeOldTryspeedTransaction(paymentID);
        }
        setCheckingPayment(false);
        setCopyTimer(0);
        const response = await post('/addpoints/generate-address', { 
            amount: data.amount, 
            game: selectedGame,
            username: data.username,
            currency: data.currency
        });

        if (response && response.errors) {
            setFormErrors(response.props.errors);
        } else {
            if (response?.data?.paymentMethodOptions) {
                setPaymentMethodOptions(response?.data?.paymentMethodOptions);                
            } else {
                //setFormErrors((prev) => ({ ...prev, general: "No payment method options available3." }));
            }
        }
    };*/

    /*seEffect(() => {
        if (data.currency && selectedGateway === 'tryspeed' && targetCurrency === data.currency) {
            handleCurrencySelect();
        }
    }, [data.currency, selectedGateway, targetCurrency]);*/

    const handlePaykenPayNow = async () => {
        try {
            setProcessing(true);
            const requestData = {
                amount: data.amount,
                game: selectedGame,
                username: data.username
            };

            const response = await post('/addpoints/payken-create', requestData);

            if (response?.props?.errors?.error) {
                setFormErrors({ general: response.props.errors.error });
                toast.error(response.props.errors.error);
                return;
            }

            // The redirect will be handled by the backend
        } catch (error) {
            console.error('Payken error:', error);
            const errorMessage = error?.response?.data?.error || 'Failed to process payment request';
            setFormErrors({ general: errorMessage });
            toast.error(errorMessage);
        } finally {
            setProcessing(false);
        }
    };

    const handleGGPayoutWalletSelect = async (walletId) => {
        // Just set the wallet in form data
        setData('wallet', walletId);
    };

    const handleGGPayoutPayNow = async () => {
        try {
            if (!data.wallet) {
                toast.error('Please select a payment method first');
                return;
            }

            // Validate minimum amount
            const amountNum = Number(data.amount);
            const minAmount = Number(settings.min_amount);
            if (amountNum < minAmount) {
                setFormErrors((prev) => ({ ...prev, amount: `Amount must be at least ${settings.min_amount} ${settings.default_currency}.` }));
                return;
            }

            // Create request data with current form data
            const requestData = {
                amount: data.amount,
                game: selectedGame,
                username: data.username,
                wallet: data.wallet
            };

            const response = await post('/addpoints/ggpayout-create', requestData);

            // Check for errors in the response
            if (response?.props?.errors?.error) {
                setFormErrors({ general: response.props.errors.error });
                toast.error(response.props.errors.error);
                return;
            }

            if (response?.data?.payment_url) {
                window.location.href = response.data.cashierUrl;
            } else {
                //const errorMessage = 'Failed to create GGPayout checkout session';
                //setFormErrors({ general: errorMessage });
                //toast.error(errorMessage);
            }
        } catch (error) {
            console.error('GGPayout error:', error);
            const errorMessage = error?.response?.data?.error || 'Failed to process payment request';
            setFormErrors({ general: errorMessage });
            toast.error(errorMessage);
        }
    };

    const PaymentTimer = () => {
        // Clear any existing timer
        if (timerRef.current) {
            clearInterval(timerRef.current);
        }

        setCopyTimer(300);
        setCheckingPayment(true);
        setMessage('Checking payment status...');

        timerRef.current = setInterval(() => {
            setCopyTimer((prev) => {
                if (prev <= 1) {
                    clearInterval(timerRef.current);
                    setCheckingPayment(false);
                    setMessage('Time expired. Please check your wallet if you already made the payment.');
                    window.location.reload();
                    return 0;
                }
                return prev - 1;
            });
        }, 1000);
    };

    const handleCopyAddress = (address) => {
        navigator.clipboard.writeText(address).then(() => {
            // Reset and start the timer
            PaymentTimer();
        });
    };

    const removeOldTryspeedTransaction = async (paymentId) => {
        try {
            const response = await axios.post('/addpoints/remove', {
                paymentId: paymentId
            });
            const data = await response.json();
                setPaymentID(null);
                clearInterval(timerRef.current);
        } catch (error) {
            console.error("Error checking payment status", error);
        }
    };

    const checkPaymentStatus = async (paymentId) => {
        try {
            const response = await fetch(`/addpoints/status/${paymentId}`);
            const data = await response.json();
            //console.log(data.status);
            if (data.status == 'paid') {
                clearInterval(timerRef.current);
                setPaymentCompleted(true);
                window.location.href = `/addpoints/success/${paymentId}`;
            } else {
                setPaymentCompleted(false);
            }
        } catch (error) {
            console.error("Error checking payment status", error);
        }
    };

    const handleGameSelect = (gameId) => {
        setSelectedGame(gameId);
        setData('game', gameId);
    };

    const handleOpenWallet = (address, walletUrl = '') => {
        if (lastAttemptedWallet !== walletUrl) {
            setWalletOpened(false);
        }

        if (walletOpened && lastAttemptedWallet === walletUrl) {
            return;
        }

        // Format the base URL based on payment type
        const baseUrl = activeTab === 'lightning' 
            ? `lightning:${address}`
            : activeTab === 'ethereum'
            ? `ethereum:${address}`
            : activeTab === 'tron'
            ? `tron:${address}` 
            : `bitcoin:${address}`;

        // If there's a custom wallet URL, use it as a prefix
        const finalUrl = walletUrl 
            ? `${walletUrl}${encodeURIComponent(baseUrl)}`
            : baseUrl;
        
        setWalletOpened(true);
        setLastAttemptedWallet(walletUrl);

        // Try to open the URL
        try {
            window.location.href = finalUrl;
        } catch (error) {
            console.error('Failed to open wallet:', error);
            // Fallback: copy to clipboard
            navigator.clipboard.writeText(address).then(() => {
                toast.success('Address copied to clipboard!');
            }).catch(err => {
                console.error('Failed to copy address:', err);
                toast.error('Failed to open wallet. Please copy the address manually.');
            });
        }
        
        if (!checkingPayment) {
            setCheckingPayment(true);
            PaymentTimer();
        }
    };

    return (
        <AuthenticatedLayout user={auth.user} settings={settings}>
            <Head title="Add Points | VegasFish" />
            <div className="flex items-center justify-center min-h-[80vh]">
                <div className="container mx-auto px-4 sm:px-6 lg:px-8 py-5 max-w-100 lg:max-w-[1020px] md:max-w-[700px]">
                    <div className="w-full p-4 py-[50px] bg-gray-100 dark:bg-gray-800 rounded-lg shadow-md">
                        <div className="text-center">
                            <h1 className="text-[28px] md:text-[35px] font-bold mb-4 inline-block font-bold bg-gradient-to-r from-[#FF6F00] to-[#960018] bg-clip-text text-transparent">Experience gaming like never before</h1>
                        </div>
                        {selectedGateway && (
                            <div className="text-center mb-6">
                                <button
                                    onClick={() => {
                                        if(paymentID) {
                                            removeOldTryspeedTransaction(paymentID);
                                            setPaymentID(null)
                                        }
                                        setData('amount', baseAmount);
                                        setSelectedGateway(false);
                                        setPaymentMethodOptions({});
                                    }}
                                    className="inline-flex items-center px-4 py-2 bg-[#FF6F00] dark:bg-[#960018] border border-[#FF6F00] dark:border-[#960018] rounded-md font-semibold text-xs text-[#fff] dark:text-[#960018] uppercase tracking-widest shadow-sm hover:bg-[#960018] dark:hover:bg-[#FF6F00] hover:border-[#960018] focus:outline-none focus:ring-2 focus:ring-[#FF6F00] focus:ring-offset-2 transition ease-in-out duration-300"
                                >
                                    <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4 mr-2" viewBox="0 0 20 20" fill="currentColor">
                                        <path fillRule="evenodd" d="M9.707 16.707a1 1 0 01-1.414 0l-6-6a1 1 0 010-1.414l6-6a1 1 0 011.414 1.414L5.414 9H17a1 1 0 110 2H5.414l4.293 4.293a1 1 0 010 1.414z" clipRule="evenodd" />
                                    </svg>
                                    Change Payment Method
                                </button>
                            </div>
                        )}
                        {!showPaymentGateways ? (
                            // Initial form
                            <form onSubmit={(e) => e.preventDefault()}>
                                <div className="mb-4">
                                    <label className="block text-md font-semibold text-gray-700 dark:text-gray-300">
                                        Enter Amount ({settings.default_currency})
                                    </label>
                                    <input
                                        type="number"
                                        value={data.amount}
                                        onChange={(e) => {
                                            setData('amount', e.target.value);
                                            setData('baseAmount', e.target.value);
                                            setbaseAmount(e.target.value);
                                        }}
                                        className="w-full p-2 mt-1 border rounded-md dark:bg-gray-700 dark:text-white"
                                        placeholder="Enter amount"
                                        min={settings.min_amount}
                                        max={settings.max_amount}
                                        step="1"
                                    />
                                    {formErrors.amount && (
                                        <div className="text-red-500 text-sm">{formErrors.amount}</div>
                                    )}
                                    {errors.amount && (
                                        <div className="text-red-500 text-sm">{errors.amount}</div>
                                    )}
                                </div>

                                <div className="mb-4">
                                    <label className="block text-md font-semibold text-gray-700 dark:text-gray-300">Enter Username</label>
                                    <input
                                        type="text"
                                        value={data.username}
                                        onChange={(e) => setData('username', e.target.value)}
                                        className="w-full p-2 mt-1 border rounded-md dark:bg-gray-700 dark:text-white"
                                        placeholder="Enter your game username"
                                    />
                                    {formErrors.username && <div className="text-red-500 text-sm">{formErrors.username}</div>}
                                </div>

                                <div className="mb-4">
                                    <label className="block text-md font-semibold text-gray-700 dark:text-gray-300">Click below to select the Game</label>
                                    <div className="grid grid-cols-4 sm:grid-cols-6 md:grid-cols-6 lg:grid-cols-8 gap-3 md:gap-5 mt-2">
                                        {games.map((game) => (
                                            <div
                                                key={game.id}
                                                onClick={() => handleGameSelect(game.id)}
                                                className={`overflow-hidden border-2 shadow-md bg-white dark:bg-gray-700 cursor-pointer rounded-[15px] ${selectedGame === game.id ? 'border-[#FF6F00]' : 'border-transparent dark:border-transparent'} hover:border-[#FF6F00]`}
                                            >
                                                {game.game_image && (
                                                    <img
                                                        src={`storage/${game.game_image}`}
                                                        alt={game.game_name}
                                                        className="w-full h-[90px] object-cover"
                                                    />
                                                )}
                                            </div>
                                        ))}
                                    </div>
                                    {formErrors.game && <div className="text-red-500 text-sm">{formErrors.game}</div>}
                                </div>
                                <div className="flex justify-center">
                                    <SwipeableButton
                                        onSuccess={handleProceed}
                                        text="Swipe to proceed!"
                                        text_unlocked="Processing..."
                                        sliderColor="#960018"
                                        sliderTextColor="#fff"
                                        sliderIconColor="#fff"
                                        textColor="#000"
                                        background_color="#FF6F00"
                                        circle
                                        name="react-swipeable-button"
                                        ref={swipeableButtonRef}
                                    />
                                </div>
                            </form>
                        ) : !selectedGateway ? (
                            // Payment Gateway Selection
                            <div className="text-center">
                                <h2 className="text-2xl font-semibold mb-6">Select Payment Method</h2>
                                {/*settings.payment_gateway_notes && (
                                    <div
                                    style={{ width: 'max-content' }}
                                    className="mx-auto p-4 md:p-5 bg-indigo-50 border border-indigo-200 rounded-2xl shadow-sm flex items-start gap-4 mb-8"
                                    >
                                    <div>
                                        <p className="text-sm md:text-base font-medium text-indigo-700 leading-snug">
                                        {settings.payment_gateway_notes}
                                        </p>
                                    </div>
                                    </div>
                                )*/}
                                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                                    {settings.enable_onchain || settings.enable_lightning ? (
                                        <button
                                            onClick={() => {
                                                if (settings.bitcoin_fees) {
                                                    setSelectedGateway('tryspeed');                                                    
                                                    const fee = parseFloat(settings.bitcoin_fees);
                                                    const amount = parseFloat(data.amount);
                                                    const amountWithFees = amount + (amount * fee / 100);
                                                    setData('amount', amountWithFees.toFixed(2));
                                                    setShowBitcoinFees(true);
                                                } else {
                                                    handleGatewaySelect('tryspeed');
                                                }
                                            }}
                                            className="p-8 bg-white dark:bg-gray-700 rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 h-full flex flex-col items-center justify-start transform hover:-translate-y-1"
                                        >
                                            <div className="w-16 h-16 mb-4 bg-[#FF6F00] dark:bg-[#236070] rounded-full flex items-center justify-center">
                                                <svg xmlns="http://www.w3.org/2000/svg" className="h-8 w-8 text-[#fff] dark:text-[#fff]" viewBox="0 0 512 512">
                                                    <path d="M504 256c0 136.967-111.033 248-248 248S8 392.967 8 256 119.033 8 256 8s248 111.033 248 248zm-141.651-35.33c4.937-32.999-20.191-50.739-54.55-62.573l11.146-44.702-27.213-6.781-10.851 43.524c-7.154-1.783-14.502-3.464-21.803-5.13l10.929-43.81-27.198-6.781-11.153 44.686c-5.922-1.349-11.735-2.682-17.377-4.084l.031-.14-37.53-9.37-7.239 29.062s20.191 4.627 19.765 4.913c11.022 2.751 13.014 10.044 12.68 15.825l-12.696 50.925c.76.194 1.744.473 2.829.907-.907-.225-1.876-.473-2.876-.713l-17.796 71.338c-1.349 3.348-4.767 8.37-12.471 6.464.271.395-19.78-4.937-19.78-4.937l-13.51 31.147 35.414 8.827c6.588 1.651 13.045 3.379 19.4 5.006l-11.262 45.213 27.182 6.781 11.153-44.733a1038.209 1038.209 0 0 0 21.687 5.627l-11.115 44.523 27.213 6.781 11.262-45.128c46.404 8.781 81.299 5.239 95.986-36.727 11.836-33.79-.589-53.281-25.004-65.991 17.78-4.098 31.174-15.792 34.747-39.949zm-62.177 87.179c-8.41 33.79-65.308 15.523-83.755 10.943l14.944-59.899c18.446 4.603 77.6 13.717 68.811 48.956zm8.417-87.667c-7.673 30.736-55.031 15.12-70.393 11.292l13.548-54.327c15.363 3.828 64.836 10.973 56.845 43.035z" fill="currentColor"/>
                                                </svg>
                                            </div>
                                            <h3 className="text-2xl font-bold mb-2 bg-gradient-to-r from-[#FF6F00] to-[#960018] bg-clip-text text-transparent">
                                                Bitcoin
                                                {settings.bitcoin_gateway_notes && (
                                                    <p className="mt-2 text-sm text-emerald-500 font-medium">
                                                        {settings.bitcoin_gateway_notes}
                                                    </p>
                                                )}
                                            </h3>
                                            <p className="text-gray-600 dark:text-gray-300 text-center text-sm">
                                                Pay using bitcoin with PayPal, Cash App and any other app!
                                            </p>
                                        </button>
                                    ) : null}
                                    {settings.enable_usdt_payment && (settings.enable_ethereum_payment || settings.enable_tron_payment) ? (
                                        <button
                                            onClick={() => {
                                                if (settings.usdt_fees) {
                                                    setSelectedGateway('usdt');                                                    
                                                    const fee = parseFloat(settings.usdt_fees);
                                                    const amount = parseFloat(data.amount);
                                                    const amountWithFees = amount + (amount * fee / 100);
                                                    setData('amount', amountWithFees.toFixed(2));
                                                    setShowBitcoinFees(true);
                                                } else {
                                                    handleGatewaySelect('usdt');
                                                }
                                            }}
                                            className="p-8 bg-white dark:bg-gray-700 rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 h-full flex flex-col items-center justify-start transform hover:-translate-y-1"
                                        >
                                            <div className="w-16 h-16 mb-4 bg-[#FF6F00] dark:bg-[#236070] rounded-full flex items-center justify-center">
                                                <svg xmlns="http://www.w3.org/2000/svg" className="h-8 w-8 text-[#fff] dark:text-[#fff]" viewBox="0 0 2000 2000">
                                                    <path d="M1000,0c552.26,0,1000,447.74,1000,1000S1552.26,2000,1000,2000,0,1552.26,0,1000,447.74,0,1000,0" fill="#53ae94"/>
                                                    <path d="M1123.42,866.76V718H1463.6V491.34H537.28V718H877.5V866.64C601,879.34,393.1,934.1,393.1,999.7s208,120.36,484.4,133.14v476.5h246V1132.8c276-12.74,483.48-67.46,483.48-133s-207.48-120.26-483.48-133m0,225.64v-0.12c-6.94.44-42.6,2.58-122,2.58-63.48,0-108.14-1.8-123.88-2.62v0.2C633.34,1081.66,451,1039.12,451,988.22S633.36,894.84,877.62,884V1050.1c16,1.1,61.76,3.8,124.92,3.8,75.86,0,114-3.16,121-3.8V884c243.8,10.86,425.72,53.44,425.72,104.16s-182,93.32-425.72,104.18" fill="#fff"/>
                                                </svg>
                                            </div>
                                            <h3 className="text-2xl font-bold mb-2 bg-gradient-to-r from-[#FF6F00] to-[#960018] bg-clip-text text-transparent">
                                                USDT
                                                {settings.usdt_gateway_notes && (
                                                    <p className="mt-2 text-sm text-emerald-500 font-medium">
                                                        {settings.usdt_gateway_notes}
                                                    </p>
                                                )}
                                            </h3>
                                            <p className="text-gray-600 dark:text-gray-300 text-center text-sm">
                                                Pay using USDT with Coinbase and Moonpay!
                                            </p>
                                        </button>
                                    ) : null}
                                    {settings.enable_usdc_payment && (
                                        <button
                                            onClick={() => handleGatewaySelect('usdc')}
                                            className="p-8 bg-white dark:bg-gray-700 rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 h-full flex flex-col items-center justify-start transform hover:-translate-y-1"
                                        >
                                            <div className="w-16 h-16 mb-4 bg-[#FF6F00] dark:bg-[#236070] rounded-full flex items-center justify-center">
                                                <svg xmlns="http://www.w3.org/2000/svg" className="h-8 w-8 text-[#fff] dark:text-[#fff]" viewBox="0 0 512 512">
                                                    <path d="M384 256c0 35.3-28.7 64-64 64s-64-28.7-64-64s28.7-64 64-64s64 28.7 64 64zM192 352a128 128 0 1 0 0-256a128 128 0 0 0 0 256zM96 256a128 128 0 1 0 256 0a128 128 0 0 0-256 0zM64 256a256 256 0 1 1 512 0A256 256 0 0 1 64 256zM0 256a256 256 0 1 1 512 0A256 256 0 0 1 0 256z" fill="currentColor"/>
                                                </svg>
                                            </div>
                                            <h3 className="text-2xl font-bold mb-2 bg-gradient-to-r from-[#FF6F00] to-[#960018] bg-clip-text text-transparent">
                                                USDC
                                                {settings.usdc_gateway_notes && (
                                                    <p className="mt-2 text-sm text-emerald-500 font-medium">
                                                        {settings.usdc_gateway_notes}
                                                    </p>
                                                )}
                                            </h3>
                                            <p className="text-gray-600 dark:text-gray-300 text-center text-sm">
                                                Pay using USDC with Coinbase!
                                            </p>
                                        </button>
                                    )}
                                    {settings.enable_ggpayout &&
                                        <button
                                            onClick={() => handleGatewaySelect('ggpayout')}
                                            className="p-8 bg-white dark:bg-gray-700 rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 h-full flex flex-col items-center justify-start transform hover:-translate-y-1"
                                        >
                                            <div className="w-16 h-16 mb-4 bg-[#FF6F00] dark:bg-[#236070] rounded-full flex items-center justify-center">
                                                <svg xmlns="http://www.w3.org/2000/svg" className="h-8 w-8 text-[#fff] dark:text-[#fff]" viewBox="0 0 576 512">
                                                    <path d="M64 64C28.7 64 0 92.7 0 128V384c0 35.3 28.7 64 64 64H512c35.3 0 64-28.7 64-64V128c0-35.3-28.7-64-64-64H64zm64 320H64V128h64V384zm96 0V128h96V384H224zm160 0V128h96V384H384zm128 0h-64V128h64V384z" fill="currentColor"/>
                                                </svg>
                                            </div>
                                            <h3 className="text-2xl font-bold mb-2 bg-gradient-to-r from-[#FF6F00] to-[#960018] bg-clip-text text-transparent">
                                                CashApp
                                                {settings.ggpayout_gateway_notes && (
                                                    <p className="mt-2 text-sm text-emerald-500 font-medium">
                                                        {settings.ggpayout_gateway_notes}
                                                    </p>
                                                )}
                                            </h3>
                                            <p className="text-gray-600 dark:text-gray-300 text-center text-sm">
                                                Pay with Cashapp
                                            </p>
                                        </button>
                                    }
                                    {/*settings.enable_payken &&
                                        <button
                                            onClick={() => handleGatewaySelect('payken')}
                                            className="p-8 bg-white dark:bg-gray-700 rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 h-full flex flex-col items-center justify-start transform hover:-translate-y-1"
                                        >
                                            <div className="w-16 h-16 mb-4 bg-[#FF6F00] dark:bg-[#236070] rounded-full flex items-center justify-center">
                                                <svg xmlns="http://www.w3.org/2000/svg" className="h-8 w-8 text-[#fff] dark:text-[#fff]" viewBox="0 0 576 512">
                                                    <path d="M64 32C28.7 32 0 60.7 0 96v32H576V96c0-35.3-28.7-64-64-64H64zM576 224H0V416c0 35.3 28.7 64 64 64H512c35.3 0 64-28.7 64-64V224zM112 352h64c8.8 0 16 7.2 16 16s-7.2 16-16 16H112c-8.8 0-16-7.2-16-16s7.2-16 16-16zm112 16c0-8.8 7.2-16 16-16H368c8.8 0 16 7.2 16 16s-7.2 16-16 16H240c-8.8 0-16-7.2-16-16z" fill="currentColor"/>
                                                </svg>
                                            </div>
                                            <h3 className="text-2xl font-bold mb-2 bg-gradient-to-r from-[#FF6F00] to-[#960018] bg-clip-text text-transparent">
                                                Payken
                                                {settings.payken_gateway_notes && (
                                                    <p className="mt-2 text-sm text-emerald-500 font-medium">
                                                        {settings.payken_gateway_notes}
                                                    </p>
                                                )}
                                            </h3>
                                            <p className="text-gray-600 dark:text-gray-300 text-center text-sm">
                                                Pay with Debit/Credit/Apple Pay/Google Pay
                                            </p>
                                        </button>
                                    */}
                                </div>
                            </div>
                        ) : selectedGateway === 'ggpayout' ? (
                            // GGPayout Wallet Selection
                            <div className="text-center">
                                <h2 className="text-2xl font-semibold mb-6">Select Payment Method</h2>
                                {formErrors.general && (
                                    <div className="mb-4 p-3 bg-red-100 border border-red-400 text-red-700 rounded">
                                        {formErrors.general}
                                    </div>
                                )}

                                {settings.cashapp_fees && (
                                    <div className="mb-6">
                                        <div className="flex flex-col items-center gap-2">
                                            <label className="block text-md font-semibold text-gray-700 dark:text-gray-300">
                                                Base Amount
                                            </label>
                                            <input
                                                type="number"
                                                value={baseAmount || ''}
                                                min={settings.min_amount}
                                                max={settings.max_amount}
                                                onChange={(e) => {
                                                    const newbaseAmount = e.target.value;
                                                    const withFees = newbaseAmount * (1 + settings.cashapp_fees/100);
                                                    setbaseAmount(newbaseAmount);
                                                    setData('amount', withFees.toFixed(2));
                                                }}
                                                className="w-64 p-2 border rounded-md dark:bg-gray-700 dark:text-white text-center"
                                                placeholder="Enter base amount"
                                            />
                                            {formErrors.amount && (
                                                <div className="text-red-500 text-sm">{formErrors.amount}</div>
                                            )}
                                            {errors.amount && (
                                                <div className="text-red-500 text-sm">{errors.amount}</div>
                                            )}
                                            <div className="p-5 rounded-2xl bg-gradient-to-r from-yellow-50 to-yellow-100 dark:from-yellow-900 dark:to-yellow-800 border border-yellow-200 dark:border-yellow-700 shadow-lg">
                                                <div className="flex justify-between items-center mb-1">
                                                    <span className="text-base md:text-lg font-medium text-yellow-900 dark:text-yellow-200">
                                                    Final amount with <span className="font-semibold">{settings.cashapp_fees}%</span> fees:
                                                    </span>
                                                    <span className="text-2xl font-extrabold text-yellow-800 dark:text-yellow-100 tracking-tight">
                                                    ${data.amount || '0.00'}
                                                    </span>
                                                </div>
                                                {settings.cashapp_fees_note && (
                                                    <p className="mt-2 text-sm text-gray-700 dark:text-gray-300 italic leading-snug">
                                                    {settings.cashapp_fees_note}
                                                    </p>
                                                )}
                                            </div>
                                        </div>
                                    </div>
                                )}

                                <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
                                    {ggPayoutWallets.map((wallet) => (
                                        <button
                                            key={wallet.id}
                                            onClick={() => !wallet.disabled && handleGGPayoutWalletSelect(wallet.id)}
                                            disabled={wallet.disabled}
                                            className={`p-4 bg-white dark:bg-[#236070] rounded-lg shadow-md transition-shadow flex flex-col items-center ${
                                                data.wallet === wallet.id ? 'ring-2 ring-[#FF6F00]' : ''
                                            } ${
                                                wallet.disabled 
                                                    ? 'opacity-50 cursor-not-allowed' 
                                                    : 'hover:shadow-lg cursor-pointer'
                                            }`}
                                        >
                                            <img 
                                                src={wallet.icon} 
                                                alt={wallet.name} 
                                                className={`w-12 h-12 mb-2 object-contain ${
                                                    wallet.disabled ? 'grayscale' : ''
                                                }`}
                                            />
                                            <span className="font-medium text-sm">{wallet.name}</span>
                                        </button>
                                    ))}
                                </div>
                                {errors.wallet && (
                                    <div className="text-red-500 text-sm mt-2">{errors.wallet}</div>
                                )}
                                
                                {/* Pay Now Button */}
                                <button
                                    onClick={handleGGPayoutPayNow}
                                    disabled={!data.wallet || processing}
                                    className={`mt-6 px-8 py-3 rounded-md text-white text-lg font-semibold transition-all duration-300 ${
                                        !data.wallet || processing
                                            ? 'bg-[#FF6F00] cursor-not-allowed'
                                            : 'bg-[#FF6F00] hover:bg-[#960018] dark:hover:bg-[#FF6F00] hover:border-[#960018]'
                                    }`}
                                >
                                    {processing ? 'Processing...' : 'Pay Now'}
                                </button>
                            </div>
                        ) : selectedGateway === 'payken' ? (
                            <div>
                                <h2 className="md:text-[35px] font-semibold text-center text-gray-800 dark:text-gray-200">
                                    Redirecting to Payken...
                                </h2>
                            </div>
                        ) : (
                            <div>
                            {showBitcoinFees && Object.keys(paymentMethodOptions).length === 0 && (
                                <div className='flex flex-col items-center gap-2'>
                                    <label className="block text-md font-semibold text-gray-700 dark:text-gray-300">
                                        Base Amount
                                    </label>
                                    <input
                                        type="number"
                                        value={baseAmount || ''}
                                        min={settings.min_amount} 
                                        max={settings.max_amount}
                                        onChange={(e) => {
                                            const newbaseAmount = e.target.value;
                                            let withFees;
                                            if (selectedGateway === 'tryspeed' && settings.bitcoin_fees) {
                                                withFees = newbaseAmount * (1 + settings.bitcoin_fees/100);
                                            } else if (selectedGateway === 'usdt' && settings.usdt_fees) {
                                                withFees = newbaseAmount * (1 + settings.usdt_fees/100);
                                            } else if (selectedGateway === 'usdc' && settings.usdc_fees) {
                                                withFees = newbaseAmount * (1 + settings.usdc_fees/100);
                                            } else {
                                                withFees = newbaseAmount;
                                            }
                                            setbaseAmount(newbaseAmount);
                                            setData('amount', withFees.toFixed(2));
                                        }}
                                        className="w-64 p-2 border rounded-md dark:bg-gray-700 dark:text-white text-center"
                                        placeholder="Enter base amount"
                                    />
                                    {formErrors.amount && (
                                        <div className="text-red-500 text-sm">{formErrors.amount}</div>
                                    )}
                                    {errors.amount && (
                                        <div className="text-red-500 text-sm">{errors.amount}</div>
                                    )}
                                    <div className="p-5 rounded-2xl bg-yellow-100 dark:bg-yellow-900 border border-yellow-200 dark:border-yellow-700 shadow-lg">
                                        <div className="flex justify-between items-center mb-1">
                                            <span className="text-base md:text-lg font-medium text-yellow-900 dark:text-yellow-200">
                                            Final amount with&nbsp;
                                            <span className="font-semibold">
                                                {selectedGateway === 'tryspeed' ? settings.bitcoin_fees :
                                                selectedGateway === 'usdt' ? settings.usdt_fees :
                                                selectedGateway === 'usdc' ? settings.usdc_fees : 0}%
                                            </span>
                                            &nbsp;fees:
                                            </span>
                                            <span className="text-2xl font-extrabold text-yellow-800 dark:text-yellow-100 tracking-tight">
                                            ${data.amount || '0.00'}
                                            </span>
                                        </div>
                                        {(selectedGateway === 'tryspeed' && settings.bitcoin_fees_note) && (
                                            <p className="mt-2 text-sm text-gray-700 dark:text-gray-300 italic leading-snug">
                                            {settings.bitcoin_fees_note}
                                            </p>
                                        )}
                                        {(selectedGateway === 'usdt' && settings.usdt_fees_note) && (
                                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                                {settings.usdt_fees_note}
                                            </p>
                                        )}
                                        {(selectedGateway === 'usdc' && settings.usdc_fees_note) && (
                                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                                {settings.usdc_fees_note}
                                            </p>
                                        )}
                                    </div>
                                    
                                    <button
                                        onClick={() => handleGatewaySelect(selectedGateway === 'tryspeed' ? 'tryspeed' : selectedGateway === 'usdt' ? 'usdt' : 'usdc')}
                                        className="mt-4 w-auto bg-[#FF6F00] hover:bg-[#960018] dark:hover:bg-[#FF6F00] hover:border-[#960018] text-white font-bold py-3 px-6 rounded-lg transition duration-300"
                                    >
                                        Proceed with Payment
                                    </button>
                                </div>
                            )}
                            

                                {Object.keys(paymentMethodOptions).length > 0  && (
                                    <>
                                    <div>
                                        <h2 className="md:text-[35px] font-semibold text-center text-gray-800 dark:text-gray-200">
                                            Invoice Generated!
                                        </h2>
                                        <p className="text-gray-600 dark:text-gray-400 text-center mt-2">
                                            Welcome to VegasFish. Please select your preferred payment method and follow the instructions to complete your payment.
                                        </p>
                                    </div>

                                    <div className="mt-4 p-4 border rounded bg-gray-100 dark:bg-gray-700">
                                        <p className="text-gray-700 dark:text-gray-300 font-bold text-lg text-center">
                                            Invoice ID: {TransactionData?.invoice_id}
                                        </p>
                                    </div>

                                    <div className="mt-4">
                                        <div className="flex flex-wrap space-x-4 justify-center">
                                            {settings.enable_lightning && paymentMethodOptions.lightning && (
                                                <button
                                                    onClick={() => setActiveTab('lightning')}
                                                    className={`py-2 px-4 rounded ${activeTab === 'lightning' ? 'bg-[#FF6F00] text-white' : 'bg-gray-200'}`}
                                                >
                                                    Lightning
                                                </button>
                                            )}
                                            {settings.enable_onchain && paymentMethodOptions.on_chain && (
                                                <button
                                                    onClick={() => setActiveTab('on_chain')}
                                                    className={`py-2 px-4 rounded ${activeTab === 'on_chain' ? 'bg-[#FF6F00] text-white' : 'bg-gray-200'}`}
                                                >
                                                    On-Chain
                                                </button>
                                            )}
                                            {paymentMethodOptions.ethereum && (
                                                <button
                                                    onClick={() => setActiveTab('ethereum')}
                                                    className={`py-2 px-4 rounded ${activeTab === 'ethereum' ? 'bg-[#FF6F00] text-white' : 'bg-gray-200'}`}
                                                >
                                                    Ethereum (ERC20)
                                                </button>
                                            )}
                                            {paymentMethodOptions.tron && (
                                                <button
                                                    onClick={() => setActiveTab('tron')}
                                                    className={`py-2 px-4 rounded ${activeTab === 'tron' ? 'bg-[#FF6F00] text-white' : 'bg-gray-200'}`}
                                                >
                                                    Tron (TRC20)
                                                </button>
                                            )}
                                        </div>

                                        {settings.lightning_payment_note && (
                                            <div className="mt-4 p-4 bg-yellow-50 dark:bg-yellow-900 border border-yellow-200 dark:border-yellow-800 rounded-lg">
                                                <p 
                                                    className="text-sm text-yellow-700 dark:text-yellow-300"
                                                    dangerouslySetInnerHTML={{ __html: settings.lightning_payment_note }}
                                                />
                                            </div>
                                        )}

                                        <div className="mt-4">
                                            {activeTab === 'lightning' && paymentMethodOptions.lightning && (
                                                <div className="flex flex-wrap gap-4 justify-center">
                                                    <button 
                                                        onClick={() => handleCopyAddress(paymentMethodOptions.lightning.payment_request)} 
                                                        className="bg-[#FF6F00] text-white font-bold py-2 rounded hover:bg-[#960018] px-4"
                                                    >
                                                        Copy Lightning Invoice
                                                    </button>
                                                    <button 
                                                        onClick={() => handleOpenWallet(paymentMethodOptions.lightning.payment_request)} 
                                                        className="bg-[#FF6F00] text-white font-bold py-2 rounded hover:bg-[#960018] px-4"
                                                    >
                                                        Open Wallet
                                                    </button>
                                                    <div className="w-full flex flex-wrap gap-4 justify-center items-center">
                                                        {getWalletButtons('lightning')
                                                            .filter(button => button.is_enabled)
                                                            .sort((a, b) => a.display_order - b.display_order)
                                                            .map((wallet) => (
                                                                <button 
                                                                    key={wallet.id}
                                                                    onClick={() => handleOpenWallet(
                                                                        paymentMethodOptions.lightning.payment_request,
                                                                        wallet.wallet_url
                                                                    )} 
                                                                    className="bg-gray-100 text-gray-800 font-semibold p-2 rounded hover:bg-gray-200 flex flex-col items-center"
                                                                >
                                                                    <img src={`/storage/${wallet.icon_url}`} alt={wallet.name} className="w-10" />
                                                                    <span className="mt-1 text-sm">{wallet.name}</span>
                                                                </button>
                                                            ))}
                                                    </div>
                                                </div>
                                            )}

                                            {activeTab === 'on_chain' && paymentMethodOptions.on_chain && (
                                                <div className="flex flex-wrap gap-4 justify-center">
                                                    <button 
                                                        onClick={() => handleCopyAddress(paymentMethodOptions.on_chain.address)} 
                                                        className="bg-[#FF6F00] text-white font-bold py-2 rounded hover:bg-[#960018] px-4"
                                                    >
                                                        Copy On-Chain Address
                                                    </button>
                                                    <button 
                                                        onClick={() => handleOpenWallet(paymentMethodOptions.on_chain.address)} 
                                                        className="bg-[#FF6F00] text-white font-bold py-2 rounded hover:bg-[#960018] px-4"
                                                    >
                                                        Open Wallet
                                                    </button>
                                                    <div className="w-full flex flex-wrap gap-4 justify-center items-center">
                                                        {getWalletButtons('onchain')
                                                            .filter(button => button.is_enabled)
                                                            .sort((a, b) => a.display_order - b.display_order)
                                                            .map((wallet) => (
                                                                <button 
                                                                    key={wallet.id}
                                                                    onClick={() => handleOpenWallet(
                                                                        paymentMethodOptions.on_chain.address,
                                                                        wallet.wallet_url
                                                                    )} 
                                                                    className="bg-gray-100 text-gray-800 font-semibold p-2 rounded hover:bg-gray-200 flex flex-col items-center"
                                                                >
                                                                    <img src={`/storage/${wallet.icon_url}`} alt={wallet.name} className="w-10" />
                                                                    <span className="mt-1 text-sm">{wallet.name}</span>
                                                                </button>
                                                            ))}
                                                    </div>
                                                </div>
                                            )}

                                            {activeTab === 'ethereum' && paymentMethodOptions.ethereum && (
                                                <div className="flex flex-wrap gap-4 justify-center">
                                                    <button 
                                                        onClick={() => handleCopyAddress(paymentMethodOptions.ethereum.address)} 
                                                        className="bg-[#FF6F00] text-white font-bold py-2 rounded hover:bg-[#960018] px-4"
                                                    >
                                                        Copy Ethereum Address
                                                    </button>
                                                    <button 
                                                        onClick={() => handleOpenWallet(paymentMethodOptions.ethereum.address)} 
                                                        className="bg-[#FF6F00] text-white font-bold py-2 rounded hover:bg-[#960018] px-4"
                                                    >
                                                        Open Wallet
                                                    </button>
                                                </div>
                                            )}

                                            {activeTab === 'tron' && paymentMethodOptions.tron && (
                                                <div className="flex flex-wrap gap-4 justify-center">
                                                    <button 
                                                        onClick={() => handleCopyAddress(paymentMethodOptions.tron.address)} 
                                                        className="bg-[#FF6F00] text-white font-bold py-2 rounded hover:bg-[#960018] px-4"
                                                    >
                                                        Copy Tron Address
                                                    </button>
                                                    <button 
                                                        onClick={() => handleOpenWallet(paymentMethodOptions.tron.address)} 
                                                        className="bg-[#FF6F00] text-white font-bold py-2 rounded hover:bg-[#960018] px-4"
                                                    >
                                                        Open Wallet
                                                    </button>
                                                </div>
                                            )}
                                        </div>

                                        {copyTimer > 0 && <p className="mt-4 text-center text-gray-700 dark:text-gray-300">Checking payment in {copyTimer}s...</p>}
                                        {copyTimer === 0 && message && <p className="text-red-500 mt-2">{message}</p>}
                                    </div>
                                    </>)}
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </AuthenticatedLayout>
    );
};

export default PaymentForm;
