import React, { useState } from 'react';
import { Head, useForm, router } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { DragDropContext, Droppable, Draggable } from '@hello-pangea/dnd';
import { Dialog } from '@headlessui/react';
import WalletButtonForm from './WalletButtonForm';
import WalletButtonItem from './WalletButtonItem';
import toast from 'react-hot-toast';

export default function WalletButtons({ walletButtons }) {
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [editingButton, setEditingButton] = useState(null);
    const [buttons, setButtons] = useState(walletButtons);

    const form = useForm({
        name: '',
        icon_url: null,
        wallet_url: '',
        type: 'both',
        is_enabled: true,
        note: '',
        display_order: 0
    });

    const openModal = (button = null) => {
        form.clearErrors();
        form.reset();

        const defaultData = {
            name: '',
            icon_url: null,
            wallet_url: '',
            type: 'both',
            is_enabled: true,
            note: '',
            display_order: buttons.length + 1
        };

        if (button) {
            const buttonData = {
                name: button.name,
                wallet_url: button.wallet_url,
                type: button.type,
                is_enabled: button.is_enabled ? true : false,
                display_order: button.display_order,
                icon_url: button.icon_url,
                note: button.note
            };
            form.setData(buttonData);
            setEditingButton(button);
        } else {
            form.setData(defaultData);
            setEditingButton(null);
        }

        setIsModalOpen(true);
    };

    const closeModal = () => {
        setIsModalOpen(false);
        setEditingButton(null);
        form.reset();
        form.clearErrors();
    };

    const handleSubmit = async (e) => {
        e.preventDefault();
        
        // Clear any existing errors
        form.clearErrors();

        // Validate required fields
        const requiredFields = ['name', 'wallet_url', 'type', 'display_order'];
        const missingFields = requiredFields.filter(field => !form.data[field]);
        
        if (missingFields.length > 0) {
            missingFields.forEach(field => {
                form.setError(field, 'This field is required');
            });
            toast.error('Please fill in all required fields');
            return;
        }

        try {
            const formData = new FormData();
            // Add common fields
            formData.append('name', form.data.name);
            formData.append('wallet_url', form.data.wallet_url);
            formData.append('type', form.data.type);
            formData.append('display_order', form.data.display_order);
            formData.append('is_enabled', form.data.is_enabled ? '1' : '0');
            formData.append('note', form.data.note);
            
            if (editingButton) {
                formData.append('_method', 'PUT');
                // Only append icon if it's a new file
                if (form.data.icon_url instanceof File) {
                    formData.append('icon_url', form.data.icon_url);
                }

                await router.post(route('admin.wallet-buttons.update', editingButton.id), formData, {
                    forceFormData: true,
                    onSuccess: () => {
                        // Visit the index page to get fresh data
                        router.visit(route('admin.wallet-buttons.index'), {
                            onSuccess: () => {
                                toast.success('Wallet button updated successfully');
                                closeModal();
                            }
                        });
                    },
                    onError: (errors) => {
                        console.error('Update errors:', errors);
                        Object.keys(errors).forEach(key => {
                            form.setError(key, errors[key]);
                        });
                        toast.error('Please correct the errors in the form');
                    }
                });
            } else {
                if (!form.data.icon_url) {
                    form.setError('icon_url', 'Icon is required for new wallet buttons');
                    toast.error('Please upload an icon');
                    return;
                }

                formData.append('icon_url', form.data.icon_url);

                await router.post(route('admin.wallet-buttons.store'), formData, {
                    forceFormData: true,
                    onSuccess: () => {
                        // Visit the index page to get fresh data
                        router.visit(route('admin.wallet-buttons.index'), {
                            onSuccess: () => {
                                toast.success('Wallet button created successfully');
                                closeModal();
                            }
                        });
                    },
                    onError: (errors) => {
                        console.error('Creation errors:', errors);
                        Object.keys(errors).forEach(key => {
                            form.setError(key, errors[key]);
                        });
                        toast.error('Please correct the errors in the form');
                    }
                });
            }
        } catch (error) {
            console.error('Operation failed:', error);
            toast.error('An unexpected error occurred. Please try again.');
        }
    };

    const handleDragEnd = async (result) => {
        if (!result.destination) return;

        const items = Array.from(buttons);
        const [reorderedItem] = items.splice(result.source.index, 1);
        items.splice(result.destination.index, 0, reorderedItem);

        const updatedButtons = items.map((item, index) => ({
            ...item,
            display_order: index + 1
        }));

        // Optimistically update the UI
        setButtons(updatedButtons);

        try {
            // Add _method field for Laravel method spoofing
            const formData = new FormData();
            formData.append('_method', 'PUT');
            formData.append('buttons', JSON.stringify(updatedButtons));

            await router.post(route('admin.wallet-buttons.update-order'), formData, {
                forceFormData: true,
                preserveScroll: true,
                preserveState: true,
                onSuccess: () => {
                    toast.success('Order updated successfully');
                },
                onError: (error) => {
                    console.error('Order update failed:', error);
                    // Revert to previous state on error
                    setButtons(buttons);
                    toast.error('Failed to update order');
                }
            });
        } catch (error) {
            console.error('Order update failed:', error);
            // Revert to previous state on error
            setButtons(buttons);
            toast.error('Failed to update order');
        }
    };

    const toggleStatus = async (button) => {
        try {
            await router.put(route('admin.wallet-buttons.toggle-status', button.id), {}, {
                preserveState: true,
                onSuccess: () => {
                    const updatedButtons = buttons.map(b => 
                        b.id === button.id ? { ...b, is_enabled: !b.is_enabled } : b
                    );
                    setButtons(updatedButtons);
                },
            });
        } catch (error) {
            console.error('Status toggle failed:', error);
        }
    };

    const deleteButton = async (button) => {
        if (!confirm('Are you sure you want to delete this wallet button?')) return;

        try {
            await router.delete(route('admin.wallet-buttons.destroy', button.id), {
                onSuccess: () => {
                    setButtons(buttons.filter(b => b.id !== button.id));
                },
            });
        } catch (error) {
            console.error('Delete failed:', error);
        }
    };

    return (
        <AdminLayout>
            <Head title="Manage Wallet Buttons" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            <div className="flex flex-col sm:flex-row justify-between items-center gap-4 sm:gap-0 mb-6">
                                <h2 className="text-xl sm:text-2xl font-semibold text-gray-900 dark:text-gray-100 text-center sm:text-left">
                                    Manage Wallet Buttons
                                </h2>
                                <button
                                    onClick={() => openModal()}
                                    className="w-full sm:w-auto px-4 py-2 bg-indigo-600 text-white text-sm sm:text-base rounded-md hover:bg-indigo-700 transition-colors"
                                >
                                    Add New Wallet Button
                                </button>
                            </div>

                            <DragDropContext onDragEnd={handleDragEnd}>
                                <Droppable droppableId="wallet-buttons">
                                    {(provided) => (
                                        <div 
                                            ref={provided.innerRef}
                                            {...provided.droppableProps}
                                            className="space-y-2"
                                        >
                                            {buttons.map((button, index) => (
                                                <Draggable
                                                    key={button.id}
                                                    draggableId={button.id.toString()}
                                                    index={index}
                                                >
                                                    {(provided) => (
                                                        <WalletButtonItem
                                                            button={button}
                                                            provided={provided}
                                                            onEdit={() => openModal(button)}
                                                            onDelete={() => deleteButton(button)}
                                                            onToggleStatus={() => toggleStatus(button)}
                                                        />
                                                    )}
                                                </Draggable>
                                            ))}
                                            {provided.placeholder}
                                        </div>
                                    )}
                                </Droppable>
                            </DragDropContext>

                            <Dialog
                                open={isModalOpen}
                                onClose={closeModal}
                                className="fixed inset-0 z-10 overflow-y-auto"
                            >
                                <div className="flex items-center justify-center min-h-screen px-4">
                                    <Dialog.Overlay className="fixed inset-0 bg-black/30" />

                                    <div className="relative bg-white dark:bg-gray-800 rounded-lg p-8 max-w-md w-full mx-auto">
                                        <Dialog.Title className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4">
                                            {editingButton ? 'Edit Wallet Button' : 'Add New Wallet Button'}
                                        </Dialog.Title>

                                        <WalletButtonForm
                                            form={form}
                                            onSubmit={handleSubmit}
                                            onCancel={closeModal}
                                            isEditing={!!editingButton}
                                        />
                                    </div>
                                </div>
                            </Dialog>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}