import React, { useState, useEffect } from 'react';
import axios from 'axios';
import AdminLayout from '../../Layouts/AdminLayout';
import { Switch } from '@headlessui/react';

const Settings = () => {
    const [settings, setSettings] = useState({
        enable_lightning_payment: true,
        enable_onchain_payment: true
    });
    const [loading, setLoading] = useState(true);
    const [saving, setSaving] = useState(false);

    useEffect(() => {
        fetchSettings();
    }, []);

    const fetchSettings = async () => {
        try {
            const response = await axios.get('/api/admin/settings/payment-methods');
            setSettings({
                enable_lightning_payment: response.data.enable_lightning_payment === 'true',
                enable_onchain_payment: response.data.enable_onchain_payment === 'true'
            });
            setLoading(false);
        } catch (error) {
            console.error('Error fetching settings:', error);
            setLoading(false);
        }
    };

    const handleToggle = async (key) => {
        const newSettings = {
            ...settings,
            [key]: !settings[key]
        };
        setSettings(newSettings);

        try {
            setSaving(true);
            await axios.post('/api/admin/settings/payment-methods', {
                [key]: newSettings[key].toString()
            });
        } catch (error) {
            console.error('Error saving settings:', error);
            // Revert the change if save failed
            setSettings(settings);
        } finally {
            setSaving(false);
        }
    };

    if (loading) {
        return (
            <AdminLayout>
                <div className="p-6">
                    <div className="animate-pulse">
                        <div className="h-4 bg-gray-200 rounded w-1/4"></div>
                        <div className="space-y-3 mt-4">
                            <div className="h-8 bg-gray-200 rounded"></div>
                            <div className="h-8 bg-gray-200 rounded"></div>
                        </div>
                    </div>
                </div>
            </AdminLayout>
        );
    }

    return (
        <AdminLayout>
            <div className="p-6">
                <h2 className="text-2xl font-semibold text-gray-900 mb-6">
                    Payment Method Settings
                </h2>

                <div className="bg-white rounded-lg shadow">
                    <div className="p-6 space-y-6">
                        {/* Lightning Network Payment Toggle */}
                        <div className="flex items-center justify-between">
                            <div>
                                <h3 className="text-lg font-medium text-gray-900">
                                    Lightning Network Payment
                                </h3>
                                <p className="text-sm text-gray-500">
                                    Enable or disable Lightning Network payment method
                                </p>
                            </div>
                            <Switch
                                checked={settings.enable_lightning_payment}
                                onChange={() => handleToggle('enable_lightning_payment')}
                                className={`${settings.enable_lightning_payment ? 'bg-indigo-600' : 'bg-gray-200'}
                                    relative inline-flex h-6 w-11 flex-shrink-0 cursor-pointer rounded-full border-2 border-transparent transition-colors duration-200 ease-in-out focus:outline-none focus:ring-2 focus:ring-indigo-600 focus:ring-offset-2`}
                                disabled={saving}
                            >
                                <span className="sr-only">Enable Lightning Network payment</span>
                                <span
                                    aria-hidden="true"
                                    className={`${settings.enable_lightning_payment ? 'translate-x-5' : 'translate-x-0'}
                                        pointer-events-none inline-block h-5 w-5 transform rounded-full bg-white shadow ring-0 transition duration-200 ease-in-out`}
                                />
                            </Switch>
                        </div>

                        {/* Bitcoin On-Chain Payment Toggle */}
                        <div className="flex items-center justify-between pt-6 border-t border-gray-200">
                            <div>
                                <h3 className="text-lg font-medium text-gray-900">
                                    Bitcoin On-Chain Payment
                                </h3>
                                <p className="text-sm text-gray-500">
                                    Enable or disable Bitcoin On-Chain payment method
                                </p>
                            </div>
                            <Switch
                                checked={settings.enable_onchain_payment}
                                onChange={() => handleToggle('enable_onchain_payment')}
                                className={`${settings.enable_onchain_payment ? 'bg-indigo-600' : 'bg-gray-200'}
                                    relative inline-flex h-6 w-11 flex-shrink-0 cursor-pointer rounded-full border-2 border-transparent transition-colors duration-200 ease-in-out focus:outline-none focus:ring-2 focus:ring-indigo-600 focus:ring-offset-2`}
                                disabled={saving}
                            >
                                <span className="sr-only">Enable Bitcoin On-Chain payment</span>
                                <span
                                    aria-hidden="true"
                                    className={`${settings.enable_onchain_payment ? 'translate-x-5' : 'translate-x-0'}
                                        pointer-events-none inline-block h-5 w-5 transform rounded-full bg-white shadow ring-0 transition duration-200 ease-in-out`}
                                />
                            </Switch>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Settings;