import { Head, Link } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { useState } from 'react';

export default function PromosIndex({ promos }) {
    const [searchTerm, setSearchTerm] = useState('');

    const filteredPromos = promos.filter(promo =>
        promo.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
        promo.description.toLowerCase().includes(searchTerm.toLowerCase())
    );

    const getStatusBadge = (promo) => {
        if (!promo.is_active) {
            return <span className="px-2 py-1 text-xs font-semibold bg-gray-100 text-gray-800 rounded-full">Inactive</span>;
        }
        
        const now = new Date();
        const startDate = promo.start_date ? new Date(promo.start_date) : null;
        const endDate = promo.end_date ? new Date(promo.end_date) : null;
        
        if (startDate && now < startDate) {
            return <span className="px-2 py-1 text-xs font-semibold bg-yellow-100 text-yellow-800 rounded-full">Scheduled</span>;
        }
        
        if (endDate && now > endDate) {
            return <span className="px-2 py-1 text-xs font-semibold bg-red-100 text-red-800 rounded-full">Expired</span>;
        }
        
        return <span className="px-2 py-1 text-xs font-semibold bg-green-100 text-green-800 rounded-full">Active</span>;
    };

    return (
        <AdminLayout>
            <Head title="Manage Promotions" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            {/* Header */}
                            <div className="flex justify-between items-center mb-6">
                                <div>
                                    <h2 className="text-2xl font-bold text-gray-900">Manage Promotions</h2>
                                    <p className="text-gray-600 mt-1">Create and manage promotional content</p>
                                </div>
                                <Link
                                    href={route('admin.promos.create')}
                                    className="inline-flex items-center px-4 py-2 bg-indigo-600 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:bg-indigo-700 focus:bg-indigo-700 active:bg-indigo-900 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 transition ease-in-out duration-150"
                                >
                                    <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 6v6m0 0v6m0-6h6m-6 0H6" />
                                    </svg>
                                    Add New Promo
                                </Link>
                            </div>

                            {/* Search */}
                            <div className="mb-6">
                                <input
                                    type="text"
                                    placeholder="Search promos..."
                                    value={searchTerm}
                                    onChange={(e) => setSearchTerm(e.target.value)}
                                    className="w-full px-4 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500"
                                />
                            </div>

                            {/* Promos Grid */}
                            {filteredPromos.length > 0 ? (
                                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                                    {filteredPromos.map((promo) => (
                                        <div key={promo.id} className="bg-white border border-gray-200 rounded-lg overflow-hidden shadow-sm hover:shadow-md transition-shadow">
                                            {/* Image */}
                                            <div className="h-48 overflow-hidden">
                                                                                            <img
                                                src={promo.image_path ? `/storage/${promo.image_path}` : '/placeholder-image.jpg'}
                                                alt={promo.image_alt || promo.title}
                                                className="w-full h-full object-cover"
                                            />
                                            </div>

                                            {/* Content */}
                                            <div className="p-4">
                                                <div className="flex justify-between items-start mb-2">
                                                    <h3 className="text-lg font-semibold text-gray-900 truncate">
                                                        {promo.title}
                                                    </h3>
                                                    {getStatusBadge(promo)}
                                                </div>

                                                <p className="text-gray-600 text-sm mb-3 line-clamp-2">
                                                    {promo.description.replace(/<[^>]*>/g, '').substring(0, 100)}...
                                                </p>

                                                {/* Date Range */}
                                                {(promo.start_date || promo.end_date) && (
                                                    <div className="text-xs text-gray-500 mb-3">
                                                        {promo.start_date && promo.end_date 
                                                            ? `${new Date(promo.start_date).toLocaleDateString()} - ${new Date(promo.end_date).toLocaleDateString()}`
                                                            : promo.start_date 
                                                                ? `From ${new Date(promo.start_date).toLocaleDateString()}`
                                                                : `Until ${new Date(promo.end_date).toLocaleDateString()}`
                                                        }
                                                    </div>
                                                )}

                                                {/* Actions */}
                                                <div className="flex space-x-2">
                                                    <Link
                                                        href={route('admin.promos.show', promo.id)}
                                                        className="flex-1 text-center px-3 py-1 text-xs bg-blue-100 text-blue-700 rounded hover:bg-blue-200 transition-colors"
                                                    >
                                                        View
                                                    </Link>
                                                    <Link
                                                        href={route('admin.promos.edit', promo.id)}
                                                        className="flex-1 text-center px-3 py-1 text-xs bg-yellow-100 text-yellow-700 rounded hover:bg-yellow-200 transition-colors"
                                                    >
                                                        Edit
                                                    </Link>
                                                    <Link
                                                        href={route('admin.promos.destroy', promo.id)}
                                                        method="delete"
                                                        as="button"
                                                        className="flex-1 text-center px-3 py-1 text-xs bg-red-100 text-red-700 rounded hover:bg-red-200 transition-colors"
                                                        onClick={(e) => {
                                                            if (!confirm('Are you sure you want to delete this promo?')) {
                                                                e.preventDefault();
                                                            }
                                                        }}
                                                    >
                                                        Delete
                                                    </Link>
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            ) : (
                                <div className="text-center py-12">
                                    <div className="bg-gray-50 rounded-lg p-8">
                                        <svg className="w-12 h-12 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v13m0-13V6a2 2 0 112 2h-2zm0 0V5.5A2.5 2.5 0 109.5 8H12zm-7 4h14M5 12a2 2 0 110-4h14a2 2 0 110 4M5 12v7a2 2 0 002 2h10a2 2 0 002-2v-7" />
                                        </svg>
                                        <h3 className="text-lg font-medium text-gray-900 mb-2">
                                            {searchTerm ? 'No promos found' : 'No promos yet'}
                                        </h3>
                                        <p className="text-gray-600 mb-4">
                                            {searchTerm ? 'Try adjusting your search terms.' : 'Get started by creating your first promotional content.'}
                                        </p>
                                        {!searchTerm && (
                                            <Link
                                                href={route('admin.promos.create')}
                                                className="inline-flex items-center px-4 py-2 bg-indigo-600 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:bg-indigo-700 focus:bg-indigo-700 active:bg-indigo-900 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 transition ease-in-out duration-150"
                                            >
                                                Create First Promo
                                            </Link>
                                        )}
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>

            <style jsx>{`
                .line-clamp-2 {
                    display: -webkit-box;
                    -webkit-line-clamp: 2;
                    -webkit-box-orient: vertical;
                    overflow: hidden;
                }
            `}</style>
        </AdminLayout>
    );
} 