import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link } from '@inertiajs/react';
import { useState } from 'react';
import toast from 'react-hot-toast';

const LogShow = ({ log }) => {
    const [isContextExpanded, setIsContextExpanded] = useState(false);

    const getLogTypeColor = (type) => {
        switch (type) {
            case 'error': return 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400';
            case 'warning': return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400';
            case 'info': return 'bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400';
            default: return 'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300';
        }
    };

    const getLogTypeIcon = (type) => {
        switch (type) {
            case 'error': return 'fas fa-exclamation-triangle';
            case 'warning': return 'fas fa-exclamation-circle';
            case 'info': return 'fas fa-info-circle';
            default: return 'fas fa-circle';
        }
    };

    return (
        <AdminLayout>
            <Head title={`Log #${log.id} - ${log.type.toUpperCase()}`} />
            <div className="py-12">
                <div className="max-w-4xl mx-auto sm:px-6 lg:px-8 space-y-6">
                    {/* Header */}
                    <div className="bg-gradient-to-r from-[#d34a40] to-[#b03a30] dark:from-[#73100d] dark:to-[#590c0a] overflow-hidden shadow-lg rounded-[15px]">
                        <div className="p-6 text-white">
                            <div className="flex items-center justify-between">
                                <div>
                                    <h1 className="text-3xl font-bold mb-2">Log Entry #{log.id}</h1>
                                    <p className="text-gray-100 dark:text-gray-200">
                                        Detailed view of system log entry
                                    </p>
                                </div>
                                <Link
                                    href={route('admin.logs.index')}
                                    className="flex items-center px-4 py-2 bg-white/20 text-white rounded-lg hover:bg-white/30 transition-colors"
                                >
                                    <i className="fas fa-arrow-left mr-2"></i>
                                    Back to Logs
                                </Link>
                            </div>
                        </div>
                    </div>

                    {/* Log Details */}
                    <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
                        {/* Log Header */}
                        <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                            <div className="flex items-center justify-between">
                                <div className="flex items-center space-x-4">
                                    <span className={`px-3 py-1 rounded-full text-sm font-medium ${getLogTypeColor(log.type)}`}>
                                        <i className={`${getLogTypeIcon(log.type)} mr-2`}></i>
                                        {log.type.toUpperCase()}
                                    </span>
                                    <span className="text-sm text-gray-600 dark:text-gray-400">
                                        {log.formatted_created_at}
                                    </span>
                                </div>
                                <div className="text-sm text-gray-600 dark:text-gray-400">
                                    ID: {log.id}
                                </div>
                            </div>
                        </div>

                        {/* Log Content */}
                        <div className="p-6 space-y-6">
                            {/* Basic Information */}
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">Basic Information</h3>
                                    <div className="space-y-3">
                                        <div className="flex justify-between">
                                            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">Source:</span>
                                            <span className="text-sm text-gray-900 dark:text-white">{log.source}</span>
                                        </div>
                                        <div className="flex justify-between">
                                            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">Type:</span>
                                            <span className={`text-sm font-medium ${getLogTypeColor(log.type)}`}>
                                                {log.type.toUpperCase()}
                                            </span>
                                        </div>
                                        <div className="flex justify-between">
                                            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">Created:</span>
                                            <span className="text-sm text-gray-900 dark:text-white">{log.formatted_created_at}</span>
                                        </div>
                                        <div className="flex justify-between">
                                            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">Updated:</span>
                                            <span className="text-sm text-gray-900 dark:text-white">{log.updated_at}</span>
                                        </div>
                                    </div>
                                </div>

                                <div>
                                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">User Information</h3>
                                    <div className="space-y-3">
                                        <div className="flex justify-between">
                                            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">User:</span>
                                            <span className="text-sm text-gray-900 dark:text-white">{log.user_name}</span>
                                        </div>
                                        {log.user_email && (
                                            <div className="flex justify-between">
                                                <span className="text-sm font-medium text-gray-700 dark:text-gray-300">Email:</span>
                                                <span className="text-sm text-gray-900 dark:text-white">{log.user_email}</span>
                                            </div>
                                        )}
                                        <div className="flex justify-between">
                                            <span className="text-sm font-medium text-gray-700 dark:text-gray-300">IP Address:</span>
                                            <span className="text-sm text-gray-900 dark:text-white">{log.ip_address || 'N/A'}</span>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {/* Message */}
                            <div>
                                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">Message</h3>
                                <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                                    <p className="text-gray-900 dark:text-white whitespace-pre-wrap">{log.message}</p>
                                </div>
                            </div>

                            {/* Context Data */}
                            {log.context && (
                                <div>
                                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">Context Data</h3>
                                    <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                                        <div className="flex items-center justify-between mb-2">
                                            <div className="flex items-center space-x-2">
                                                <span className="text-sm font-medium text-gray-700 dark:text-gray-300">JSON Format</span>
                                                <span className="text-xs text-gray-500 dark:text-gray-400">
                                                    ({log.context_count} items)
                                                </span>
                                            </div>
                                            <div className="flex items-center space-x-2">
                                                <button
                                                    onClick={() => setIsContextExpanded(!isContextExpanded)}
                                                    className="text-xs bg-gray-600 text-white px-2 py-1 rounded hover:bg-gray-700 transition-colors"
                                                >
                                                    {isContextExpanded ? 'Collapse' : 'Expand'}
                                                </button>
                                                <button
                                                    onClick={() => {
                                                        navigator.clipboard.writeText(log.formatted_context);
                                                        toast.success('JSON copied to clipboard!');
                                                    }}
                                                    className="text-xs bg-blue-600 text-white px-2 py-1 rounded hover:bg-blue-700 transition-colors"
                                                >
                                                    Copy JSON
                                                </button>
                                            </div>
                                        </div>
                                        <div className={`bg-gray-900 dark:bg-gray-800 rounded border border-gray-700 p-4 overflow-x-auto transition-all duration-300 ${
                                            isContextExpanded ? 'max-h-none' : 'max-h-32'
                                        }`}>
                                            <pre className="text-sm text-green-400 dark:text-green-300 font-mono">
                                                <code>{log.formatted_context}</code>
                                            </pre>
                                        </div>
                                    </div>
                                </div>
                            )}

                            {/* User Agent */}
                            {log.user_agent && (
                                <div>
                                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">User Agent</h3>
                                    <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                                        <p className="text-sm text-gray-900 dark:text-white break-all">{log.user_agent}</p>
                                    </div>
                                </div>
                            )}

                            {/* Related Information */}
                            <div>
                                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">Related Information</h3>
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <div className="bg-blue-50 dark:bg-blue-900/30 rounded-lg p-4">
                                        <div className="flex items-center">
                                            <i className="fas fa-user-circle text-blue-600 dark:text-blue-400 mr-3"></i>
                                            <div>
                                                <p className="text-sm font-medium text-blue-900 dark:text-blue-300">User</p>
                                                <p className="text-sm text-blue-700 dark:text-blue-400">{log.user_name}</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div className="bg-green-50 dark:bg-green-900/30 rounded-lg p-4">
                                        <div className="flex items-center">
                                            <i className="fas fa-server text-green-600 dark:text-green-400 mr-3"></i>
                                            <div>
                                                <p className="text-sm font-medium text-green-900 dark:text-green-300">Source</p>
                                                <p className="text-sm text-green-700 dark:text-green-400">{log.source}</p>
                                            </div>
                                        </div>
                                    </div>
                                    {log.ip_address && (
                                        <div className="bg-purple-50 dark:bg-purple-900/30 rounded-lg p-4">
                                            <div className="flex items-center">
                                                <i className="fas fa-network-wired text-purple-600 dark:text-purple-400 mr-3"></i>
                                                <div>
                                                    <p className="text-sm font-medium text-purple-900 dark:text-purple-300">IP Address</p>
                                                    <p className="text-sm text-purple-700 dark:text-purple-400">{log.ip_address}</p>
                                                </div>
                                            </div>
                                        </div>
                                    )}
                                    <div className="bg-orange-50 dark:bg-orange-900/30 rounded-lg p-4">
                                        <div className="flex items-center">
                                            <i className="fas fa-clock text-orange-600 dark:text-orange-400 mr-3"></i>
                                            <div>
                                                <p className="text-sm font-medium text-orange-900 dark:text-orange-300">Timestamp</p>
                                                <p className="text-sm text-orange-700 dark:text-orange-400">{log.formatted_created_at}</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Action Buttons */}
                    <div className="flex justify-between items-center">
                        <Link
                            href={route('admin.logs.index')}
                            className="flex items-center px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700 transition-colors"
                        >
                            <i className="fas fa-arrow-left mr-2"></i>
                            Back to Logs
                        </Link>
                        <div className="flex space-x-2">
                            <button
                                onClick={() => {
                                    const dataStr = JSON.stringify(log, null, 2);
                                    const dataBlob = new Blob([dataStr], {type: 'application/json'});
                                    const url = URL.createObjectURL(dataBlob);
                                    const link = document.createElement('a');
                                    link.href = url;
                                    link.download = `log-${log.id}-${log.type}-${new Date().toISOString().split('T')[0]}.json`;
                                    link.click();
                                }}
                                className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                            >
                                <i className="fas fa-download mr-2"></i>
                                Export JSON
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default LogShow; 