import React, { useEffect } from 'react';
import { Link, router } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import 'datatables.net-dt/css/dataTables.dataTables.css';
import $ from 'jquery';
import 'datatables.net';

const Index = ({ games }) => {
    useEffect(() => {
        // Initialize DataTable
        $('#gamesTable').DataTable({
            responsive: true,
            paging: true,
            searching: true,
            ordering: true,
            info: true,
            lengthMenu: [5, 10, 25, 50],
            pageLength: 10, // Show 10 results by default
            language: {
                lengthMenu: "Show _MENU_ entries",
                info: "Showing _START_ to _END_ of _TOTAL_ entries",
                infoEmpty: "No entries available",
                infoFiltered: "(filtered from _MAX_ total entries)",
            },
            // Use DataTable's built-in classes for styling
            dom: '<"top"l>rt<"bottom"ip><"clear">',
        });
    }, []);

    const handleDelete = (gameId) => {
        if (confirm('Are you sure you want to delete this game?')) {
            router.delete(route('admin.games.destroy', gameId), {
                onSuccess: () => {
                    // DataTable will be refreshed when the page reloads
                    $('#gamesTable').DataTable().destroy();
                }
            });
        }
    };

    const handleToggleStatus = (gameId, currentStatus) => {
        const action = currentStatus === 'active' ? 'deactivate' : 'activate';
        if (confirm(`Are you sure you want to ${action} this game?`)) {
            router.patch(route('admin.games.toggle-status', gameId), {
                onSuccess: () => {
                    // DataTable will be refreshed when the page reloads
                    $('#gamesTable').DataTable().destroy();
                }
            });
        }
    };

    return (
        <AdminLayout>
            <div className="container mx-auto p-6">
                {/* Header Section */}
                <div className="bg-gradient-to-r from-[#d34a40] to-[#b03a30] dark:from-[#73100d] dark:to-[#590c0a] p-6 rounded-[15px] shadow-lg mb-6">
                    <h1 className="text-3xl font-bold text-white">Games Management</h1>
                    <p className="text-gray-100 dark:text-gray-200 mt-2">Manage your game library and settings</p>
                </div>

                <div className="flex justify-end mb-4">
                    <Link 
                        href="/admin/games/create" 
                        className="bg-[#d34a40] text-white px-6 py-3 rounded-lg hover:bg-[#b03a30] dark:bg-[#73100d] dark:hover:bg-[#590c0a] transition-colors flex items-center shadow-md"
                    >
                        <i className="fas fa-plus mr-2"></i>
                        Add New Game
                    </Link>
                </div>

                <div className="bg-white dark:bg-gray-800 rounded-[15px] shadow-lg overflow-hidden border border-gray-200 dark:border-gray-700">
                    <div className="overflow-x-auto">
                        <table className="w-full">
                            <thead className="bg-gradient-to-r from-[#d34a40] to-[#b03a30] dark:from-[#73100d] dark:to-[#590c0a] text-white">
                                <tr>
                                    <th className="px-6 py-4 text-left">Game Name</th>
                                    <th className="px-6 py-4 text-left">Image</th>
                                    <th className="px-6 py-4 text-left">Status</th>
                                    <th className="px-6 py-4 text-left">Actions</th>
                                </tr>
                            </thead>
                            <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                                {games.map(game => (
                                    <tr key={game.id} className="hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors">
                                        <td className="px-6 py-4 text-gray-900 dark:text-gray-100">{game.game_name}</td>
                                        <td className="px-6 py-4">
                                            {game.game_image && (
                                                <img 
                                                    src={`/storage/${game.game_image}`} 
                                                    alt={game.game_name} 
                                                    className="w-20 h-20 object-cover rounded-lg shadow-md dark:opacity-90" 
                                                />
                                            )}
                                        </td>
                                        <td className="px-6 py-4">
                                            <span className={`px-3 py-1 rounded-full text-sm font-medium ${
                                                game.status === 'active' 
                                                    ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400'
                                                    : 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400'
                                            }`}>
                                                {game.status}
                                            </span>
                                        </td>
                                        <td className="px-6 py-4">
                                            <div className="flex space-x-3">
                                                <Link 
                                                    href={`/admin/games/${game.id}/edit`}
                                                    className="text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300 transition-colors"
                                                    title="Edit Game"
                                                >
                                                    <i className="fas fa-edit"></i>
                                                </Link>
                                                <button 
                                                    type="button"
                                                    className={`transition-colors ${
                                                        game.status === 'active' 
                                                            ? 'text-orange-600 dark:text-orange-400 hover:text-orange-800 dark:hover:text-orange-300'
                                                            : 'text-green-600 dark:text-green-400 hover:text-green-800 dark:hover:text-green-300'
                                                    }`}
                                                    onClick={() => handleToggleStatus(game.id, game.status)}
                                                    title={game.status === 'active' ? 'Deactivate Game' : 'Activate Game'}
                                                >
                                                    <i className={`fas ${game.status === 'active' ? 'fa-pause' : 'fa-play'}`}></i>
                                                </button>
                                                <button 
                                                    type="button"
                                                    className="text-red-600 dark:text-red-400 hover:text-red-800 dark:hover:text-red-300 transition-colors"
                                                    onClick={() => handleDelete(game.id)}
                                                    title="Delete Game"
                                                >
                                                    <i className="fas fa-trash"></i>
                                                </button>
                                                <Link 
                                                    href={`/admin/games/${game.id}`}
                                                    className="text-green-600 dark:text-green-400 hover:text-green-800 dark:hover:text-green-300 transition-colors"
                                                    title="View Game"
                                                >
                                                    <i className="fas fa-eye"></i>
                                                </Link>
                                            </div>
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Index;
