import React from 'react';
import { useForm, Link } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head } from '@inertiajs/react';
import PrimaryButton from '@/Components/PrimaryButton';
import InputLabel from '@/Components/InputLabel';
import TextInput from '@/Components/TextInput';
import InputError from '@/Components/InputError';
import { toast } from 'react-hot-toast';

const Edit = ({ game }) => {
    const { data, setData, post, processing, errors } = useForm({
        game_name: game.game_name,
        game_image: null,
        status: game.status,
        _method: 'PUT', // Add this for method spoofing
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        
        // Create FormData object to properly handle file upload
        const formData = new FormData();
        formData.append('game_name', data.game_name);
        formData.append('status', data.status);
        formData.append('_method', 'PUT');
        
        if (data.game_image) {
            formData.append('game_image', data.game_image);
        }

        // Use post instead of put for file uploads
        post(`/admin/games/${game.id}`, {
            data: formData,
            forceFormData: true,
            onSuccess: () => {
                toast.success('Game updated successfully');
            },
            onError: (errors) => {
                toast.error('Please correct the errors in the form');
            }
        });
    };

    return (
        <AdminLayout>
            <Head title="Edit Game" />
            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* Header Section */}
                    <div className="bg-gradient-to-r from-[#d34a40] to-[#b03a30] dark:from-[#73100d] dark:to-[#590c0a] p-6 rounded-[15px] shadow-lg mb-6">
                        <h1 className="text-3xl font-bold text-white">Edit Game</h1>
                        <p className="text-gray-100 dark:text-gray-200 mt-2">Modify game details and settings</p>
                    </div>

                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-lg rounded-[15px] border border-gray-200 dark:border-gray-700">
                        <div className="p-6">
                            <form onSubmit={handleSubmit} className="space-y-6" encType="multipart/form-data">
                                <div>
                                    <InputLabel htmlFor="game_name" value="Game Name" className="text-gray-700 dark:text-gray-300" />
                                    <TextInput
                                        id="game_name"
                                        type="text"
                                        value={data.game_name}
                                        className="mt-1 block w-full bg-gray-50 dark:bg-gray-900 border border-gray-300 dark:border-gray-700 text-gray-900 dark:text-gray-100"
                                        onChange={(e) => setData('game_name', e.target.value)}
                                        required
                                    />
                                    <InputError message={errors.game_name} className="mt-2" />
                                </div>

                                <div>
                                    <InputLabel htmlFor="game_image" value="Game Image" className="text-gray-700 dark:text-gray-300" />
                                    {game.game_image && (
                                        <div className="mb-4">
                                            <img 
                                                src={`/storage/${game.game_image}`} 
                                                alt={game.game_name} 
                                                className="w-32 h-32 object-cover rounded-lg shadow-md border border-gray-200 dark:border-gray-700 dark:opacity-90" 
                                            />
                                        </div>
                                    )}
                                    <input
                                        type="file"
                                        id="game_image"
                                        onChange={(e) => setData('game_image', e.target.files[0])}
                                        className="mt-1 block w-full text-gray-700 dark:text-gray-300 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:bg-[#d34a40] dark:file:bg-[#73100d] file:text-white hover:file:bg-[#b03a30] dark:hover:file:bg-[#590c0a] file:transition-colors"
                                    />
                                    <InputError message={errors.game_image} className="mt-2" />
                                </div>

                                <div>
                                    <InputLabel htmlFor="status" value="Status" className="text-gray-700 dark:text-gray-300" />
                                    <select
                                        id="status"
                                        value={data.status}
                                        onChange={(e) => setData('status', e.target.value)}
                                        className="mt-1 block w-full bg-gray-50 dark:bg-gray-900 border border-gray-300 dark:border-gray-700 text-gray-900 dark:text-gray-100 rounded-lg focus:ring-[#d34a40] focus:border-[#d34a40] dark:focus:ring-[#73100d] dark:focus:border-[#73100d]"
                                    >
                                        <option value="active">Active</option>
                                        <option value="inactive">Inactive</option>
                                    </select>
                                    <InputError message={errors.status} className="mt-2" />
                                </div>

                                <div className="flex justify-end gap-4">
                                    <Link
                                        href={route('admin.games.index')}
                                        className="inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-700 rounded-lg text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                                    >
                                        Cancel
                                    </Link>
                                    <PrimaryButton 
                                        disabled={processing}
                                        className="bg-[#d34a40] hover:bg-[#b03a30] dark:bg-[#73100d] dark:hover:bg-[#590c0a]"
                                    >
                                        Update Game
                                    </PrimaryButton>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Edit;
