import React, { useState, useEffect } from 'react';
import { Link, usePage } from '@inertiajs/react';
import ThemeToggle from '@/Components/ThemeToggle';
import ResponsiveNavLink from '@/Components/ResponsiveNavLink';
import { Toaster } from 'react-hot-toast';
import ApplicationLogo from '@/Components/ApplicationLogo';
import Dropdown from '@/Components/Dropdown';
import NavLink from '@/Components/NavLink';

const AdminLayout = ({ children }) => {
    const [isSidebarOpen, setIsSidebarOpen] = useState(true);
    const [isGamesSubmenuOpen, setIsGamesSubmenuOpen] = useState(false);
    const [isProfileSubmenuOpen, setIsProfileSubmenuOpen] = useState(false);
    const [showingNavigationDropdown, setShowingNavigationDropdown] = useState(false);
    const { auth } = usePage().props;
    const isAdmin = auth.user.role === 'admin';
    const isCashier = auth.user.role === 'cashier';

    const toggleSidebar = () => {
        setIsSidebarOpen(!isSidebarOpen);
    };

    const toggleGamesSubmenu = () => {
        setIsGamesSubmenuOpen(!isGamesSubmenuOpen);
    };

    const toggleProfileSubmenu = () => {
        setIsProfileSubmenuOpen(!isProfileSubmenuOpen);
    };

    useEffect(() => {
        const handleResize = () => {
            if (window.innerWidth < 768) {
                setIsSidebarOpen(false);
            }
        };

        // Set initial state
        handleResize();

        // Add event listener
        window.addEventListener('resize', handleResize);

        // Cleanup
        return () => window.removeEventListener('resize', handleResize);
    }, []);

    return (
        <div className="flex flex-col min-h-screen bg-gray-50 dark:bg-gray-900">
            <Toaster
                position="top-right"
                toastOptions={{
                    duration: 4000,
                    style: {
                        background: '#333',
                        color: '#fff',
                    },
                    success: {
                        duration: 3000,
                        style: {
                            background: '#059669',
                        },
                    },
                    error: {
                        duration: 5000,
                        style: {
                            background: '#DC2626',
                        },
                    },
                }}
            />
            <header className="bg-gradient-to-r from-[#d34a40] to-[#b03a30] dark:from-[#73100d] dark:to-[#590c0a] shadow-lg z-50 fixed w-full top-0">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 flex justify-between items-center h-16">
                    <div className="flex items-center">
                        <button onClick={toggleSidebar} className="text-white hover:text-gray-200 focus:outline-none mr-5 transition-colors">
                            <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M4 6h16M4 12h16m-7 6h7" />
                            </svg>
                        </button>
                        <Link href={route('admin.dashboard')} className="text-xl font-bold text-white hover:text-gray-200 transition-colors">
                            {isAdmin ? 'Admin Panel' : 'Cashier Panel'}
                        </Link>
                    </div>
                    <div className="flex items-center space-x-4">
                        <div className="text-white hidden sm:block">
                            <span className="mr-2">Welcome,</span>
                            <span className="font-semibold">{auth.user.name}</span>
                            <span className="ml-2 px-2 py-1 text-sm rounded-full bg-white/20">
                                {auth.user.role.charAt(0).toUpperCase() + auth.user.role.slice(1)}
                            </span>
                        </div>
                        <ThemeToggle />
                        <Link
                            href={route('logout')}
                            method="post"
                            as="button"
                            className="text-white hover:text-gray-200 transition-colors flex items-center px-3 py-2 rounded-lg hover:bg-[#b03a30] dark:hover:bg-[#590c0a]"
                        >
                            <i className="fas fa-sign-out-alt mr-2"></i>
                            Logout
                        </Link>
                    </div>
                </div>
            </header>

            <div className="flex flex-1">
                <nav className={`fixed top-0 left-0 h-full bg-white dark:bg-gray-800 border-r border-gray-200 dark:border-gray-700 ${isSidebarOpen ? 'w-64' : 'w-20'} mt-16`}>
                    <div className="p-4">
                        <ul className={`space-y-2 ${isSidebarOpen ? 'open' : 'close'}`}>
                            <li>
                                <Link 
                                    href={route('admin.dashboard')}
                                    className="flex items-center px-4 py-3 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                >
                                    <i className="fas fa-tachometer-alt mr-3"></i>
                                    {isSidebarOpen && <span>Dashboard</span>}
                                </Link>
                            </li>

                            {isAdmin && (
                                <>
                                    <li>
                                        <button 
                                            onClick={toggleGamesSubmenu}
                                            className="flex items-center w-full px-4 py-3 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                        >
                                            <i className="fas fa-gamepad mr-3"></i>
                                            {isSidebarOpen && (
                                                <>
                                                    <span>Games</span>
                                                    <i className={`fas fa-chevron-down ml-auto transition-transform ${isGamesSubmenuOpen ? 'rotate-180' : ''}`}></i>
                                                </>
                                            )}
                                        </button>
                                        
                                        {isGamesSubmenuOpen && isSidebarOpen && (
                                            <ul className="mt-2 ml-4 space-y-2">
                                                <li>
                                                    <Link 
                                                        href={route('admin.games.index')}
                                                        className="flex items-center px-4 py-2 text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                                    >
                                                        <i className="fas fa-list mr-2"></i>
                                                        <span>All Games</span>
                                                    </Link>
                                                </li>
                                                <li>
                                                    <Link 
                                                        href={route('admin.games.create')}
                                                        className="flex items-center px-4 py-2 text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                                    >
                                                        <i className="fas fa-plus mr-2"></i>
                                                        <span>Add Game</span>
                                                    </Link>
                                                </li>
                                            </ul>
                                        )}
                                    </li>

                                    <li>
                                        <Link 
                                            href={route('admin.users.index')}
                                            className="flex items-center px-4 py-3 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                        >
                                            <i className="fas fa-users mr-3"></i>
                                            {isSidebarOpen && <span>Users</span>}
                                        </Link>
                                    </li>

                                    <li>
                                        <Link 
                                            href={route('admin.wallet-buttons.index')}
                                            className="flex items-center px-4 py-3 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                        >
                                            <i className="fas fa-wallet mr-3"></i>
                                            {isSidebarOpen && <span>Wallet Buttons</span>}
                                        </Link>
                                    </li>

                                    <li>
                                        <Link 
                                            href={route('admin.promos.index')}
                                            className="flex items-center px-4 py-3 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                        >
                                            <i className="fas fa-gift mr-3"></i>
                                            {isSidebarOpen && <span>Promos</span>}
                                        </Link>
                                    </li>

                                    <li>
                                        <Link 
                                            href={route('admin.settings.index')}
                                            className="flex items-center px-4 py-3 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                        >
                                            <i className="fas fa-cog mr-3"></i>
                                            {isSidebarOpen && <span>Settings</span>}
                                        </Link>
                                    </li>

                                    <li>
                                        <Link
                                            href={route('admin.wallet.index')}
                                            active={route().current('admin.wallet.index')}
                                            className="flex items-center px-4 py-3 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                        >
                                            <svg xmlns="http://www.w3.org/2000/svg" className={`h-5 w-5 ${isSidebarOpen ? 'mr-3' : ''}`} viewBox="0 0 20 20" fill="currentColor">
                                                <path d="M4 4a2 2 0 00-2 2v1h16V6a2 2 0 00-2-2H4z" />
                                                <path fillRule="evenodd" d="M18 9H2v5a2 2 0 002 2h12a2 2 0 002-2V9zM4 13a1 1 0 011-1h1a1 1 0 110 2H5a1 1 0 01-1-1zm5-1a1 1 0 100 2h1a1 1 0 100-2H9z" clipRule="evenodd" />
                                            </svg>
                                            {isSidebarOpen && <span>Wallet Balance</span>}
                                        </Link>
                                    </li>
                                </>
                            )}

                            <li>
                                <Link 
                                    href={route('admin.transactions.index')}
                                    className="flex items-center px-4 py-3 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                >
                                    <i className="fas fa-exchange-alt mr-3"></i>
                                    {isSidebarOpen && <span>Transactions</span>}
                                </Link>
                            </li>

                            
                            <li>
                                <Link 
                                    href={route('admin.transaction-stats')}
                                    className="flex items-center px-4 py-3 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                >
                                    <i className="fas fa-chart-bar mr-3"></i>
                                    {isSidebarOpen && <span>Transaction Stats</span>}
                                </Link>
                            </li>

                            <li>
                                <Link 
                                    href={route('admin.customer-transaction-stats')}
                                    className="flex items-center px-4 py-3 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                >
                                    <i className="fas fa-users-cog mr-3"></i>
                                    {isSidebarOpen && <span>Customer Stats</span>}
                                </Link>
                            </li>
                            
                            {isAdmin && (
                                <li>
                                    <Link 
                                        href={route('admin.logs.index')}
                                        className="flex items-center px-4 py-3 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                    >
                                        <i className="fas fa-file-alt mr-3"></i>
                                        {isSidebarOpen && <span>System Logs</span>}
                                    </Link>
                                </li>
                            )}

                            <li>
                                <button 
                                    onClick={toggleProfileSubmenu}
                                    className="flex items-center w-full px-4 py-3 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                >
                                    <i className="fas fa-user-circle mr-3"></i>
                                    {isSidebarOpen && (
                                        <>
                                            <span>My Profile</span>
                                            <i className={`fas fa-chevron-down ml-auto transition-transform ${isProfileSubmenuOpen ? 'rotate-180' : ''}`}></i>
                                        </>
                                    )}
                                </button>
                                
                                {isProfileSubmenuOpen && isSidebarOpen && (
                                    <ul className="mt-2 ml-4 space-y-2">
                                        <li>
                                            <Link 
                                                href={route('admin.profile.edit')}
                                                className="flex items-center px-4 py-2 text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                                            >
                                                <i className="fas fa-user-edit mr-2"></i>
                                                <span>Edit Profile</span>
                                            </Link>
                                        </li>
                                    </ul>
                                )}
                            </li>
                        </ul>
                    </div>
                </nav>

                <main className={`flex-1 p-6 overflow-x-hidden bg-gray-50 dark:bg-gray-900 ${isSidebarOpen ? 'ml-20 md:ml-64' : 'ml-20'} mt-16`}>
                    {children}
                </main>
            </div>

            <footer className={`bg-white dark:bg-gray-800 border-t border-gray-200 dark:border-gray-700 shadow-lg ${isSidebarOpen ? 'ml-20 md:ml-64' : 'ml-20'}`}>
                <div className="max-w-7xl mx-auto px-4 py-4 text-center">
                    <p className="text-gray-600 dark:text-gray-400">&copy; {new Date().getFullYear()} VegasFish. All rights reserved.</p>
                </div>
            </footer>
        </div>
    );
};

export default AdminLayout;
