<?php
namespace App\Services;

use Illuminate\Support\Facades\Auth;
use App\Models\Setting;
use App\Helpers\PayUtils; // Add this import
use App\Models\Transaction;
use App\Models\User;
use App\Models\Game;
use Telegram\Bot\Laravel\Facades\Telegram;

class PaymentService
{
    protected $apiService;
    protected $ggPayoutUrl = 'https://api.ggpayout.com/api/pay/create';
    protected $trySpeedUrl = 'https://api.tryspeed.com/payments';
    protected $trySpeedWithdrawalDecodeUrl = 'https://api.tryspeed.com/withdraws/decode';
    protected $ggPayoutFillUrl = 'https://api.ggpayout.com/api/ecashapp/fill';
    protected $paykenBaseUrl = 'https://app.payken.io/invoice';
    protected $chatId;

    public function __construct(ApiService $apiService)
    {
        $this->apiService = $apiService;
        $this->chatId = env('TELEGRAM_CHAT_ID_WITHDRAWAL');
    }

    public function processPayment($request)
    {
        $gateway = $request->input('paymentGateway', 'tryspeed');
        
        if ($gateway === 'ggpayout') {
            return $this->processGGPayoutPayment($request);
        } else if ($gateway === 'payken') {
            return $this->processPaykenPayment($request);
        }
        
        return $this->processTrySpeedPayment($request);
    }

    public function processTrySpeedPayment($request)
    {
        try {
            $defaultCurrency = Setting::get('default_currency', 'USD');
                $data = [
                    'amount' => $request->input('amount'),
                    'currency' => $defaultCurrency,
                    'target_currency' => $request->input('currency'),
                    'metadata' => [
                        'user_id' => Auth::user()->id,
                        'idn_no' => Auth::user()->idn_no,
                        'game_id' => $request->input('game'),
                        'username' => $request->input('username'),
                        'payment_from' => 'vegasfish',
                    ]
                ];
                $response = $this->apiService->post($this->trySpeedUrl, $data, [
                    'Authorization' => 'Basic ' . env('TRYSPEED_API_KEY'),
                    'accept' => 'application/json'
                ]);
                
            return $response;
        } catch (\Exception $e) {
            \Log::error('TrySpeed payment processing error', [
                'error' => $e->getMessage(),
                'request' => $request->all()
            ]);
            throw $e;
        }
    }
    
    public function WithdrawalDecode($withdraw_request)
    {
        try {
            $data = [
                'withdraw_request' => $withdraw_request,
            ];
            
            $response = $this->apiService->post($this->trySpeedWithdrawalDecodeUrl, $data, [
                'Authorization' => 'Basic ' . env('TRYSPEED_API_KEY'),
                'accept' => 'application/json'
            ]);

            if (!isset($response['amount'])) {
                return false;
            }

            // Return true if amount is 0, false otherwise
            return $response['amount'] === 0;

        } catch (\Exception $e) {
            \Log::error('TrySpeed payment processing error', [
                'error' => $e->getMessage(),
            ]);
            throw $e;
        }
    }

    public function processGGPayoutPayment($request)
    {
        try {
            $timestamp = round(microtime(true) * 1000);
            $orderNo = $this->generateOrderNumber();
            
            $data = [
                'mchNo' => env('GGPAYOUT_MERCHANT_ID'),
                'mchOrderNo' => $orderNo,
                'amount' => $request->input('amount') * 100, // Convert to cents
                'currency' => 'usd',
                'wayCode' => $request->input('wallet'),
                'clientIp' => $request->ip(),
                'wayParam' => [
                    'clientId' => Auth::user()->id,
                    'idnNo' => Auth::user()->idn_no,
                    'gameId' => $request->input('game'),
                    'username' => $request->input('username'),
                    'userId' => Auth::user()->id,
                ],
                'notifyUrl' => config('app.url') . '/api/webhooks/ggpayout',
                'returnUrl' => route('payment.ggpayout.success', ['orderId' => $orderNo]),
                'expiredTime' => 7200,
                'timestamp' => $timestamp,
                'signType' => 'MD5'
            ];

            // Generate signature
            $signedData = PayUtils::sign($data, env('GGPAYOUT_API_KEY'));

            // Verify signature before sending
            if (!PayUtils::verifySign($signedData, env('GGPAYOUT_API_KEY'))) {
                throw new \Exception('Signature verification failed');
            }
            
            $response = $this->apiService->post($this->ggPayoutUrl, $signedData, [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ]);
            if (!isset($response['code']) && $response['code'] !== 0) {
                throw new \Exception($response['msg'] ?? 'Failed to create GGPayout order');
            }
            if($response['code'] !== 0) {
                return [
                    'code' => $response['code'],
                    'msg' => $response['msg']?? 'Failed to create GGPayout order',
                    'data' => [
                        'cashierUrl' => null
                    ]
                    ];
            }

            return [
                'id' => $orderNo,
                'code' => $response['code'],
                'status' => 'pending',
                'payment_url' => $response['data']['cashierUrl'] ?? null,
                'amount' => $request->input('amount'),
                'currency' => 'USD',
                'payment_method' => $request->input('wallet')
            ];

        } catch (\Exception $e) {
            \Log::error('GGPayout payment processing error', [
                'error' => $e->getMessage(),
                'request' => $request->all()
            ]);
            throw $e;
        }
    }

    public function processPaykenPayment($request)
    {
        try {
            $orderNo = $this->generateOrderNumber();
            $amount = $request->input('amount');
            $game = $request->input('game');
            $username = $request->input('username');
            $userId = Auth::user()->id;
            $email = Auth::user()->email;

            // Create transaction record first
            $transaction = Transaction::create([
                'user_id' => $userId,
                'username' => $username,
                'game_id' => $game,
                'amount' => $amount,
                'currency' => 'USD',
                'type' => 'deposit',
                'payment_status' => 'pending',
                'mchOrderNo' => $orderNo,
                'payment_method' => 'payken',
                'payment_gateway' => 'payken'
            ]);

            // Construct Payken payment URL
            $paykenUrl = sprintf(
                '%s/merchant_id=%s&merchant_secret=%s?display_name=%s&user_id=%s&total=%s&type=purchase&order_id=%s',
                $this->paykenBaseUrl,
                env('PAYKEN_MERCHANT_ID', '0747a3584e'),
                env('PAYKEN_MERCHANT_SECRET', 'a9f962e21910e7b1c31f4113'),
                urlencode("Game Points Purchase - Order #$orderNo"),
                urlencode($email),
                $amount,
                $orderNo
            );

            // Prepare response data
            return $responseData = [
                'TransactionData' => $transaction,
                'code' => 0,
                'payment_url' => $paykenUrl,
                'id' => $orderNo
            ];
            

            /*return [
                'code' => 0,
                'payment_url' => $paykenUrl,
                'id' => $orderNo
            ];*/

        } catch (\Exception $e) {
            \Log::error('Payken payment processing error', [
                'error' => $e->getMessage(),
                'request' => $request->all()
            ]);
            throw $e;
        }
    }

    public function handlePaykenCallback($request)
    {
        try {
            $status = $request->input('status');
            $orderId = $request->input('order_id');
            $transactionId = $request->input('transaction_id');
            $amount = $request->input('amount');
            $paymentType = $request->input('payment_type');

            $transaction = Transaction::where('mchOrderNo', $orderId)->first();

            if (!$transaction) {
                throw new \Exception('Transaction not found');
            }

            // Update transaction status
            $transaction->update([
                'payment_status' => strtolower($status) === 'success' ? 'paid' : 'failed',
                'transaction_id' => $transactionId,
                'payment_type' => $paymentType
            ]);

            // If payment was successful, send Telegram notification
            if (strtolower($status) === 'success') {
                $user = User::find($transaction->user_id);
                $game = Game::find($transaction->game_id);
                $amountFormatted = number_format((float)$transaction->amount, 2, '.', '');
                
                $question = "Player ID: {$user->idn_no}\n"
                    . "Player Nickname (username): {$transaction->username}\n"
                    . "Game: {$game->game_name}\n"
                    . "Total purchase: {$amountFormatted}\n"
                    . "Task Number: {$transaction->invoice_id}\n"
                    . "Payment Method: {$transaction->payment_method}\n"
                    . "Platform URL: " . config('app.url');
                
                // Send poll and store message info
                $pollResponse = Telegram::sendPoll([
                    'chat_id' => env('TELEGRAM_CHAT_ID_PAID'),
                    'question' => $question,
                    'options' => json_encode(['Accept', 'Declined']),
                    'is_anonymous' => false,
                ]);

                // Store Telegram message info in transaction
                if (isset($pollResponse['message_id'])) {
                    $transaction->update([
                        'telegram_message_id' => $pollResponse['message_id'],
                        'telegram_chat_id' => env('TELEGRAM_CHAT_ID_PAID')
                    ]);
                }

                // Add your points update logic here
                // This depends on your application's points system
            }

            return [
                'success' => true,
                'message' => 'Payment status updated successfully'
            ];

        } catch (\Exception $e) {
            \Log::error('Payken callback processing error', [
                'error' => $e->getMessage(),
                'request' => $request->all()
            ]);
            throw $e;
        }
    }

    protected function generateOrderNumber()
    {
        return 'ORD' . time() . rand(1000, 9999);
    }

    public function processWithdrawal($request)
    {
        $gateway = $request->input('gateway', 'tryspeed');
        
        if ($gateway === 'ggpayout') {
            return $this->processGGPayoutWithdrawal($request);
        }
        
        return $this->processTrySpeedWithdrawal($request);
    }

    protected function processTrySpeedWithdrawal($request)
    {
        try {
            $url = 'https://api.tryspeed.com/send';
            $defaultCurrency = Setting::get('default_currency', 'USD');
            $targetCurrency = $request->input('target_currency');
            $address = $request->input('address');
            // Check if address is lightning, onchain, Ethereum or Tron
            if(str_starts_with($address, 'lnbc')){
                $withdraw_method = 'lightning';
            }else if(str_starts_with($address, 'bc1')){
                $withdraw_method = 'onchain';
            }else if (str_starts_with($address, '0x')) {
                $withdraw_method = 'ethereum';
            }else if( str_starts_with($address, 'T')) {
                $withdraw_method = 'tron';
            }
            
            $data = [
                'amount' => $request->input('amount'),
                'currency' => $targetCurrency,
                'target_currency' => $targetCurrency,
                'withdraw_method' => $withdraw_method, //$request->input('network') === 'lightning' ? 'lightning' : 'onchain',
                'withdraw_request' => $request->input('address'),
                'note' => 'Withdrawal for transaction ID: ' . ($request->input('metadata.transaction_id') ?? 'Unknown')
            ];
            logEvent('info', 'PaymentService', 'TrySpeed withdrawal request before api calling', $data);

            $response = $this->apiService->post($url, $data, [
                'Authorization' => 'Basic ' . env('TRYSPEED_API_KEY'),
                'accept' => 'application/json',
                'content-type' => 'application/json'
            ]);
            logEvent('info', 'PaymentService', 'TrySpeed withdrawal request api response', $response);
            if (!isset($response['withdraw_id'])) {
                throw new \Exception('Invalid response from TrySpeed API');
            }

            return $response;
        } catch (\Exception $e) {
            \Log::error('TrySpeed withdrawal processing error', [
                'error' => $e->getMessage(),
                'request' => $request->all()
            ]);
            throw $e;
        }
    }

    protected function processGGPayoutWithdrawal($request)
    {
        // Implement GGPayout withdrawal if supported
        throw new \Exception('GGPayout withdrawals are not supported yet');
    }

    public function fillEcashappOrder($payOrderNo, $mchOrderNo, $identifier)
    {
        try {
            $timestamp = round(microtime(true) * 1000);
            
            $data = [
                'mchNo' => env('GGPAYOUT_MERCHANT_ID'),
                'payOrderNo' => $payOrderNo,
                'mchOrderNo' => $mchOrderNo,
                'identifier' => $identifier,
                'timestamp' => $timestamp,
                'signType' => 'MD5'
            ];

            // Generate signature
            $signedData = PayUtils::sign($data, env('GGPAYOUT_API_KEY'));

            // Verify signature before sending
            if (!PayUtils::verifySign($signedData, env('GGPAYOUT_API_KEY'))) {
                throw new \Exception('Signature verification failed');
            }
            
            $response = $this->apiService->post($this->ggPayoutFillUrl, $signedData, [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ]);
            
            if (!isset($response['code']) || $response['code'] !== 0) {
                throw new \Exception($response['msg'] ?? 'Failed to fill Ecashapp order');
            }

            return [
                'success' => true,
                'message' => $response['msg'] ?? 'Order filled successfully',
                'transaction_id' => $response['data']['payOrderNo'] ?? null,
                'state' => $response['data']['state'] ?? null
            ];

        } catch (\Exception $e) {
            \Log::error('Ecashapp fill order error', [
                'error' => $e->getMessage(),
                'payOrderNo' => $payOrderNo,
                'mchOrderNo' => $mchOrderNo
            ]);
            throw $e;
        }
    }
}
