<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Log extends Model
{
    use HasFactory;

    protected $table = 'logs';

    protected $fillable = [
        'type',        // 'error', 'info', 'warning'
        'source',      // Controller or service name
        'message',     // Log message
        'context',     // JSON data (nullable)
        'ip_address',  // Nullable, IP address of request
        'user_agent',  // Nullable, User agent of request
        'user_id'      // Nullable, Foreign key (user)
    ];

    protected $casts = [
        'context' => 'array',  // Ensure context is cast as an array
    ];

    /**
     * Relationship with User Model (if user_id exists).
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Create a new log entry.
     *
     * @param string      $type      Log type ('error', 'info', 'warning')
     * @param string      $source    Source (e.g., 'WebhookController', 'PaymentService')
     * @param string      $message   Log message
     * @param array|null  $context   Additional data (optional)
     * @param int|null    $userId    User ID (optional)
     */
    public static function createLog($type, $source, $message, $context = null, $userId = null)
    {
        if (env('ENABLE_LOGS', false)) {
            self::create([
                'type'       => $type,
                'source'     => $source,
                'message'    => $message,
                'context'    => $context,
                'ip_address' => request()->ip(),
                'user_agent' => request()->header('User-Agent'),
                'user_id'    => $userId
            ]);
        }
    }
}