<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Transaction;
use Illuminate\Support\Facades\Log;
use Telegram\Bot\Laravel\Facades\Telegram;
use Illuminate\Support\Facades\Http;
use App\Services\PaymentService;

class TelegramWebhookController extends Controller
{
    protected $paymentService;

    public function __construct(PaymentService $paymentService)
    {
        $this->paymentService = $paymentService;
    }

    public function handleWebhook(Request $request)
    {
        logEvent('info', 'TelegramWebhookController', 'Telegram webhook received', $request->all());
        $update = $request->all();

        // Handle poll update
        if (isset($update['poll'])) {
            return $this->handlePollUpdate($update['poll']);
        }

        // Handle message reply (existing logic)
        if (isset($update['message']['reply_to_message'])) {
            return $this->handleReply($update);
        }

        return response()->json(['status' => 'success']);
    }

    private function handlePollUpdate($poll)
    {
        //logEvent('info', 'TelegramWebhookController', 'Processing poll', [ 'poll_id' => $poll['id']]);

        try {
            // Get the transaction associated with this poll
            $transaction = Transaction::where('telegram_poll_id', $poll['id'])->first();

            if (!$transaction) {
                logEvent('warning', 'TelegramWebhookController', 'No transaction found for poll', [
                    'poll_id' => $poll['id']
                ]);
                return response()->json(['status' => 'error', 'message' => 'Transaction not found']);
            }

            // Stop the poll if total_voter_count is 1
            if ($poll['total_voter_count'] == 1 && $poll['is_closed'] == false) {
                Telegram::stopPoll([
                    'chat_id' => $transaction->telegram_chat_id,
                    'message_id' => $transaction->telegram_message_id
                ]);
                logEvent('info', 'TelegramWebhookController', 'Poll stopped due to total voter count being 1', [
                    'poll_id' => $poll['id']
                ]);
                return response()->json(['status' => 'success']);
            }

        } catch (\Exception $e) {
            logEvent('error', 'TelegramWebhookController', 'Error processing poll answer', [
                'error' => $e->getMessage(),
                'poll_answer' => $poll
            ]);

            return response()->json(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    private function handleReply($update)
    {
        logEvent('info', 'handleReply', 'Telegram webhook received', $update);
        $replyToMessageId = $update['message']['reply_to_message']['message_id'];
        $chatId = $update['message']['chat']['id'];
        $fromUsername = $update['message']['from']['username'] ?? 'Unknown';
        
        // Find the transaction associated with this message
        $transaction = Transaction::where('telegram_message_id', (string)$replyToMessageId)
                                 ->where('telegram_chat_id', (string)$chatId)
                                 ->first();
        
        if (!$transaction) {
            logEvent('warning', 'TelegramWebhookController', 'No transaction found for telegram message', [
                'message_id' => $replyToMessageId,
                'chat_id' => $chatId
            ]);
            return response()->json(['status' => 'error', 'message' => 'No transaction found']);
        }
        
        // Check if this is a photo (payment proof) and transaction is unpaid
        if (isset($update['message']['photo']) && $transaction->payment_status === 'unpaid') {
            return $this->handlePhotoReply($update, $transaction, $fromUsername);
        } else if (isset($update['message']['photo']) && $transaction->payment_status !== 'unpaid') {
            // Photo reply but transaction not unpaid
            /*Telegram::sendMessage([
                'chat_id' => $update['message']['chat']['id'],
                'text' => "⚠️ This transaction (#{$transaction->invoice_id}) is not in unpaid status. Current status: '{$transaction->payment_status}'",
                'reply_to_message_id' => $update['message']['message_id']
            ]);*/
            
            return response()->json(['status' => 'error', 'message' => 'Transaction not in unpaid status']);
        }
        
        // This is a text reply - just log it
        /*logEvent('info', 'TelegramWebhookController', 'Text reply received for transaction', [
            'transaction_id' => $transaction->invoice_id,
            'text' => $update['message']['text'] ?? 'No text'
        ]);*/
        
        return response()->json(['status' => 'success']);
    }
    
    private function handlePhotoReply($update, $transaction, $fromUsername)
    {
        // Get the largest photo (last in the array)
        $photos = $update['message']['photo'];
        $photo = end($photos);
        $fileId = $photo['file_id'];
        
        // Get file path from Telegram
        $response = Telegram::getFile(['file_id' => $fileId]);
        $filePath = $response['file_path'];
        $telegramBotToken = config('telegram.bot_token');
        
        // Full URL to download the file
        $fileUrl = "https://api.telegram.org/file/bot{$telegramBotToken}/{$filePath}";
        
        /*logEvent('info', 'TelegramWebhookController', 'Processing payment with photo', [
            'transaction_id' => $transaction->id,
            'file_url' => $fileUrl
        ]);*/
        
        $success = $this->processPaymentWithPaymentService($transaction, $fromUsername);

        /*if ($success === true) {
            // Send confirmation message
            Telegram::sendMessage([
                'chat_id' => $update['message']['chat']['id'],
                'text' => "✅ Payment for transaction #{$transaction->invoice_id} has been processed!\n\nAmount: {$transaction->amount} {$transaction->currency}\nUsername: {$transaction->username}\nProcessed by: @{$fromUsername}",
                'reply_to_message_id' => $update['message']['message_id']
            ]);
        } else {
            // Send error message
            Telegram::sendMessage([
                'chat_id' => $update['message']['chat']['id'],
                'text' => "❌ Error processing payment for transaction #{$transaction->invoice_id}. Please check logs or try again.",
                'reply_to_message_id' => $update['message']['message_id']
            ]);
        }*/
        
        return response()->json(['status' => $success ? 'success' : 'error']);
    }
    
    private function processPaymentWithPaymentService($transaction, $processedBy)
    {
        try {
            logEvent('info', 'TelegramWebhookController', 'Calling PaymentService for withdrawal', [
                'transaction_id' => $transaction->invoice_id,
                'address' => $transaction->payment_address,
                'amount' => $transaction->amount
            ]);
            
            $requestData = new \Illuminate\Http\Request();
            $requestData->merge([
                'amount' => $transaction->amount,
                'address' => $transaction->payment_address,
                'network' => $transaction->payment_method,
                'target_currency' => $transaction->target_currency,
                'metadata' => [
                    'transaction_id' => $transaction->invoice_id,
                    'processed_by' => $processedBy
                ]
            ]);
            
            try {
                $response = $this->paymentService->processWithdrawal($requestData);
                
                logEvent('info', 'TelegramWebhookController', 'PaymentService response', [
                    'transaction_id' => $transaction->invoice_id,
                    'response' => $response
                ]);

                // Update transaction with payment service response
                $transaction->update([
                    'payment_status' => $response['status'] ?? 'unpaid',
                    'transaction_id' => $response['withdraw_id'] ?? null,
                    'processed_at' => now(),
                    'processed_by' => $processedBy,
                    'payment_method' => $response['payment_method'] ?? $transaction->payment_method
                ]);

                return true;

            } catch (\Exception $e) {
                logEvent('error', 'TelegramWebhookController', 'Error processing withdrawal with PaymentService', [
                    'transaction_id' => $transaction->invoice_id,
                    'error' => $e->getMessage()
                ]);
                
                // Update transaction to failed status
                /*$transaction->update([
                    'payment_status' => 'failed',
                    'processed_at' => now(),
                    'processed_by' => $processedBy
                ]);*/

                return false;
            }
        } catch (\Exception $e) {
            logEvent('error', 'TelegramWebhookController', 'Error processing payment with PaymentService', [
                'transaction_id' => $transaction->invoice_id,
                'error' => $e->getMessage()
            ]);
            
            return false;
        }
    }
}
