<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Log;
use App\Models\User;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Carbon\Carbon;

class LogController extends Controller
{
    /**
     * Display a listing of logs with filtering and pagination
     */
    public function index(Request $request)
    {
        $query = Log::with('user');

        // Filter by log type
        if ($request->filled('type') && $request->type !== 'all') {
            $query->where('type', $request->type);
        }

        // Filter by source
        if ($request->filled('source') && $request->source !== 'all') {
            $query->where('source', 'like', '%' . $request->source . '%');
        }

        // Filter by user
        if ($request->filled('user_id') && $request->user_id !== 'all') {
            $query->where('user_id', $request->user_id);
        }

        // Filter by date range
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Search in message, source, IP, and context JSON
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('message', 'like', '%' . $search . '%')
                  ->orWhere('source', 'like', '%' . $search . '%')
                  ->orWhere('ip_address', 'like', '%' . $search . '%')
                  ->orWhereRaw("JSON_SEARCH(context, 'one', ?, null, '$.*')", ["%{$search}%"])
                  ->orWhereRaw("context LIKE ?", ["%{$search}%"]);
            });
        }

        // Get unique sources for filter dropdown
        $sources = Log::distinct()->pluck('source')->sort()->values();

        // Get users for filter dropdown
        $users = User::select('id', 'name')->orderBy('name')->get();

        // Get log statistics
        $stats = [
            'total_logs' => Log::count(),
            'error_logs' => Log::where('type', 'error')->count(),
            'warning_logs' => Log::where('type', 'warning')->count(),
            'info_logs' => Log::where('type', 'info')->count(),
            'today_logs' => Log::whereDate('created_at', today())->count(),
            'this_week_logs' => Log::whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])->count(),
        ];

        // Paginate results
        $perPage = $request->get('per_page', 50);
        $logs = $query->orderBy('created_at', 'desc')->paginate($perPage);

        // Add additional data to each log
        $logs->getCollection()->transform(function ($log) {
            $log->formatted_created_at = $log->created_at->format('M d, Y H:i:s');
            
            // Handle context data properly
            if ($log->context) {
                // If context is already a string (JSON), decode it first
                if (is_string($log->context)) {
                    $contextData = json_decode($log->context, true);
                    $log->formatted_context = json_encode($contextData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
                    $log->context_count = is_array($contextData) ? count($contextData) : 0;
                } else {
                    // Context is already an array (from model casting)
                    $log->formatted_context = json_encode($log->context, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
                    $log->context_count = count($log->context);
                }
            } else {
                $log->formatted_context = null;
                $log->context_count = 0;
            }
            
            $log->user_name = $log->user ? $log->user->name : 'System';
            $log->user_email = $log->user && isset($log->user->email) ? $log->user->email : null;
            return $log;
        });

        return Inertia::render('Admin/Logs/Index', [
            'logs' => $logs,
            'filters' => [
                'type' => $request->get('type', 'all'),
                'source' => $request->get('source', 'all'),
                'user_id' => $request->get('user_id', 'all'),
                'date_from' => $request->get('date_from', ''),
                'date_to' => $request->get('date_to', ''),
                'search' => $request->get('search', ''),
                'per_page' => $perPage,
            ],
            'sources' => $sources,
            'users' => $users,
            'stats' => $stats,
        ]);
    }

    /**
     * Display the specified log entry
     */
    public function show(Log $log)
    {
        $log->load('user');
        $log->formatted_created_at = $log->created_at->format('M d, Y H:i:s');
        
        // Handle context data properly
        if ($log->context) {
            // If context is already a string (JSON), decode it first
            if (is_string($log->context)) {
                $contextData = json_decode($log->context, true);
                $log->formatted_context = json_encode($contextData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
                $log->context_count = is_array($contextData) ? count($contextData) : 0;
            } else {
                // Context is already an array (from model casting)
                $log->formatted_context = json_encode($log->context, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
                $log->context_count = count($log->context);
            }
        } else {
            $log->formatted_context = null;
            $log->context_count = 0;
        }
        
        $log->user_name = $log->user ? $log->user->name : 'System';
        $log->user_email = $log->user && isset($log->user->email) ? $log->user->email : null;

        return Inertia::render('Admin/Logs/Show', [
            'log' => $log,
        ]);
    }

    /**
     * Clear old logs
     */
    public function clear(Request $request)
    {
        $request->validate([
            'days' => 'required|integer|min:1|max:365',
            'type' => 'nullable|in:error,info,warning,all',
        ]);

        $query = Log::query();

        // Filter by type if specified
        if ($request->filled('type') && $request->type !== 'all') {
            $query->where('type', $request->type);
        }

        // Delete logs older than specified days
        $cutoffDate = now()->subDays($request->days);
        $deletedCount = $query->where('created_at', '<', $cutoffDate)->delete();

        return response()->json([
            'success' => true,
            'message' => "Successfully deleted {$deletedCount} log entries older than {$request->days} days.",
            'deleted_count' => $deletedCount,
        ]);
    }

    /**
     * Export logs
     */
    public function export(Request $request)
    {
        $request->validate([
            'format' => 'required|in:json,csv',
            'type' => 'nullable|in:error,info,warning,all',
            'date_from' => 'nullable|date',
            'date_to' => 'nullable|date|after_or_equal:date_from',
        ]);

        $query = Log::with('user');

        // Apply filters
        if ($request->filled('type') && $request->type !== 'all') {
            $query->where('type', $request->type);
        }
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $logs = $query->orderBy('created_at', 'desc')->get();

        if ($request->format === 'json') {
            return response()->json($logs);
        } else {
            $filename = 'logs-' . now()->format('Y-m-d-H-i-s') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"',
            ];

            $callback = function () use ($logs) {
                $file = fopen('php://output', 'w');
                fputcsv($file, ['ID', 'Type', 'Source', 'Message', 'User', 'IP Address', 'User Agent', 'Created At']);

                foreach ($logs as $log) {
                    fputcsv($file, [
                        $log->id,
                        $log->type,
                        $log->source,
                        $log->message,
                        $log->user ? $log->user->name : 'System',
                        $log->ip_address,
                        $log->user_agent,
                        $log->created_at->format('Y-m-d H:i:s'),
                    ]);
                }

                fclose($file);
            };

            return response()->stream($callback, 200, $headers);
        }
    }

    /**
     * Get log statistics for dashboard
     */
    public function stats()
    {
        $stats = [
            'total_logs' => Log::count(),
            'error_logs' => Log::where('type', 'error')->count(),
            'warning_logs' => Log::where('type', 'warning')->count(),
            'info_logs' => Log::where('type', 'info')->count(),
            'today_logs' => Log::whereDate('created_at', today())->count(),
            'this_week_logs' => Log::whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])->count(),
            'recent_errors' => Log::where('type', 'error')
                ->where('created_at', '>=', now()->subDays(7))
                ->count(),
            'top_sources' => Log::selectRaw('source, COUNT(*) as count')
                ->groupBy('source')
                ->orderByDesc('count')
                ->limit(10)
                ->get(),
        ];

        return response()->json($stats);
    }

    /**
     * Debug method to check log data structure
     */
    public function debug()
    {
        $logs = Log::limit(5)->get();
        $debugData = [];
        
        foreach ($logs as $log) {
            $debugData[] = [
                'id' => $log->id,
                'context_type' => gettype($log->context),
                'context_raw' => $log->context,
                'context_json' => json_encode($log->context),
                'context_decoded' => is_string($log->context) ? json_decode($log->context, true) : null,
            ];
        }
        
        return response()->json($debugData);
    }
} 