<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;

class SetupTelegramWebhook extends Command
{
    protected $signature = 'telegram:setup-webhook {--url= : The webhook URL}';
    protected $description = 'Set up the Telegram webhook';

    public function handle()
    {
        $token = env('TELEGRAM_BOT_TOKEN');
        
        if (empty($token)) {
            $this->error('TELEGRAM_BOT_TOKEN not found in .env file!');
            return 1;
        }

        $url = $this->option('url') ?: url('/api/webhook/telegram');
        
        $this->info("Setting up Telegram webhook to: {$url}");
        
        try {
            $client = new Client();
            $response = $client->post("https://api.telegram.org/bot{$token}/setWebhook", [
                'json' => [
                    'url' => $url,
                    'allowed_updates' => ['message', 'poll', 'poll_answer'],
                    'drop_pending_updates' => true
                ],
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json'
                ]
            ]);
            
            $result = json_decode($response->getBody(), true);
            
            if ($result['ok'] ?? false) {
                $this->info('Webhook set up successfully!');
                $this->info('Description: ' . ($result['description'] ?? 'No description provided'));
                
                // Get webhook info to verify settings
                $infoResponse = $client->post("https://api.telegram.org/bot{$token}/getWebhookInfo");
                $webhookInfo = json_decode($infoResponse->getBody(), true);
                
                $this->info("\nWebhook Info:");
                $this->info("URL: " . ($webhookInfo['result']['url'] ?? 'Not set'));
                $this->info("Allowed Updates: " . json_encode($webhookInfo['result']['allowed_updates'] ?? []));
                
                return 0;
            }
            
            $this->error('Failed to set up webhook: ' . ($result['description'] ?? 'Unknown error'));
            return 1;
            
        } catch (RequestException $e) {
            if ($e->hasResponse()) {
                $statusCode = $e->getResponse()->getStatusCode();
                $body = json_decode($e->getResponse()->getBody(), true);
                
                $this->error("HTTP Error {$statusCode}: " . ($body['description'] ?? 'Unknown error'));
                $this->line('Full response:');
                $this->line(json_encode($body, JSON_PRETTY_PRINT));
            } else {
                $this->error('Connection error: ' . $e->getMessage());
            }
            return 1;
        } catch (\Exception $e) {
            $this->error('Unexpected error: ' . $e->getMessage());
            return 1;
        }
    }
} 